/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.lafwidget.animation.effects;

import java.awt.Component;
import java.util.HashMap;
import java.util.Map;

import javax.swing.ButtonModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jvnet.lafwidget.LafWidgetUtilities;
import org.jvnet.lafwidget.animation.*;

/**
 * Listener for the "glow" effects on icons.
 * 
 * @author Kirill Grouchnikov
 * @see FadeKind#ICON_GLOW
 */
public class IconGlowListener {
	/**
	 * Listener on the model changes.
	 */
	protected ChangeListener modelListener;

	/**
	 * The associated component.
	 */
	protected Component comp;

	/**
	 * Animation ID of the glow icon loop.
	 */
	protected long glowLoopAnimationId;

	/**
	 * The associated model.
	 */
	protected ButtonModel buttonModel;

	/**
	 * Key - {@link FadeKind}, value - {@link Boolean}
	 */
	protected Map<FadeKind, Boolean> prevStateMap;

	/**
	 * Creates a new listener on model changes that can cause fade animation
	 * transitions.
	 * 
	 * @param comp
	 * 		Component.
	 * @param buttonModel
	 * 		Model for the component.
	 * @param callback
	 * 		Optional application callback.
	 */
	public IconGlowListener(Component comp, ButtonModel buttonModel) {
		this.comp = comp;
		this.buttonModel = buttonModel;

		this.prevStateMap = new HashMap<FadeKind, Boolean>();
		this.prevStateMap.put(FadeKind.ICON_GLOW, buttonModel.isRollover());
	}

	/**
	 * Tracks a single change to the model.
	 * 
	 * @param fadeKind
	 * 		Fade animation kind.
	 * @param newState
	 * 		New value of the relevant attribute of the model.
	 */
	protected void trackModelChange(FadeKind fadeKind, boolean newState) {
		if (LafWidgetUtilities.toIgnoreAnimations(this.comp))
			return;
		try {
			if (this.prevStateMap.containsKey(fadeKind)) {
				boolean prevState = this.prevStateMap.get(fadeKind);
				if (!prevState && newState) {
					this.glowLoopAnimationId = FadeTracker.getInstance()
							.trackFadeLooping(
									fadeKind,
									LafWidgetUtilities.getAnimationKind(comp)
											.derive(0.1f), comp, null, false,
									null, -1, true);
				} else if (prevState && !newState) {
					FadeTracker.getInstance().requestStopOnCycleBreak(
							this.glowLoopAnimationId);
				}
			}
		} finally {
			this.prevStateMap.put(fadeKind, Boolean.valueOf(newState));
		}
	}

	/**
	 * Registers listeners on the relevant model changes.
	 */
	public void registerListeners() {
		this.modelListener = new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				if (FadeConfigurationManager.getInstance().fadeAllowed(
						FadeKind.ICON_GLOW, comp)) {
					trackModelChange(FadeKind.ICON_GLOW, buttonModel
							.isRollover());
				}
			}
		};
		this.buttonModel.addChangeListener(this.modelListener);
	}

	/**
	 * Unregisters all listeners on model changes.
	 */
	public void unregisterListeners() {
		this.buttonModel.removeChangeListener(this.modelListener);
	}
}
