/*
 *  @(#)ChainableExceptionHelper.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.util.throwable.v1;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Serializable;

/**
 * Helper class to support easy-to-implement chainable exceptions.  In
 * most situations, it is close to impossible to create a generic chainable
 * exception due to inheritance restrictions on the Java exception hierarchy
 * (this shows a Java weakness - exception categories should be interfaces,
 * not specific classes).
 * <P>
 * This will attempt to use the owning source's <tt>initCause()</tt> and
 * <tt>getCause()</tt> methods, provided the owning source provides those
 * methods.  Only the superclass's implementation that provides these methods
 * yet does not implement <tt>IChainableException</tt> will be used
 * (to prevent a possible recursion nightmare).
 * <P>
 * In order to prevent endless recursion, this class will not look at the
 * JDK 1.4 implementation of the source exception.  This used to work with
 * JDK 1.4.0, but JDK 1.4.2 seems to have broken the original implementation.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/05/06 05:35:01 $
 * @since      July 7, 2002
 */
public class ChainableExceptionHelper implements Serializable
{
    private Throwable source;
    
    /**
     * The chained-to cause for this exception.  Note that if the source cause
     * is set and allows the cause to be set, then this cause will be null.
     * If the source cannot store the cause, then this cause will be set to
     * the correct cause, which may be <tt>null</tt>.
     */
    private Throwable cause;
    
    /**
     * Flag to indicate if the cause was ever set.  Needed, since the set cause
     * may be <tt>null</tt>.
     */
    private boolean causeSet = false;
    
    
    /**
     * Sets the owning throwable.  The initCause() method can still be
     * called after this constructor is used.
     */
    public ChainableExceptionHelper( Throwable source )
    {
        if (source == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.source = source;
    }
    
    
    public ChainableExceptionHelper( Throwable source, Throwable cause )
    {
        this( source );
        initCause( cause );
    }
    
    
    /**
     * JDK 1.4 compatible method.
     * <P>
     * <i>from the JDK 1.4 documentation:</i>
     * <BLOCKQUOTE>
     * Returns the cause of this throwable or <tt>null</tt> if the cause is
     * nonexistent or unknown. (The cause is the throwable that caused this
     * throwable to get thrown.)
     * <P>
     * This implementation returns the cause that was supplied via one of the
     * constructors requiring a <tt>Throwable</tt>, or that was set after
     * creation with the <tt>initCause( Throwable )</tt> method. While it is
     * typically unnecessary to override this method, a subclass can override
     * it to return a cause set by some other means. This is appropriate for a
     * "legacy chained throwable" that predates the addition of chained
     * exceptions to <tt>Throwable</tt>. Note that it is not necessary to
     * override any of the <tt>PrintStackTrace</tt> methods, all of which
     * invoke the getCause method to determine the cause of a throwable.
     * </BLOCKQUOTE>
     *
     * @return the cause of this throwable or <tt>null</tt> if the cause is
     *      nonexistent or unknown.
     */
    public Throwable getCause()
    {
        Throwable t = null;
        if (this.causeSet)
        {
            // may still be null
            t = this.cause;
        }
        return t;
    }
    
    
    /**
     * JDK 1.4 compatible method.
     * <P>
     * <i>from the JDK 1.4 documentation:</i>
     * <BLOCKQUOTE>
     * Initializes the cause of this throwable to the specified value.
     * (The cause is the throwable that caused this throwable to get thrown.)
     * <P>
     * This method can be called at most once. It is generally called from
     * within the constructor, or immediately after creating the throwable.
     * If this throwable was created with Throwable(Throwable) or
     * Throwable(String,Throwable), this method cannot be called even once.
     * </BLOCKQUOTE>
     *
     * @param cause the cause (which is saved for later retrieval by the
     *      getCause() method). (A null value is permitted, and indicates
     *      that the cause is nonexistent or unknown.)
     * @param source the exception that will be the underlying exception
     * @return a reference to this Throwable instance.
     * @exception IllegalArgumentException if cause is this throwable.
     *      (A throwable cannot be its own cause.)
     * @exception IllegalStateException if this throwable was created with
     *      Throwable(Throwable) or Throwable(String,Throwable), or this
     *      method has already been called on this throwable.
     */
    public synchronized Throwable initCause( Throwable cause )
    {
        if (this.causeSet)
        {
            throw new IllegalStateException("Already set cause");
        }
        if (cause == this.source)
        {
            throw new IllegalArgumentException(
                "exception cannot cause itself." );
        }
        
        this.causeSet = true;
        this.cause = cause;
        return this.source;
    }
    
    
    /**
     * For non-JDK 1.4 compatible VMs, this overrides the original behavior
     * to describe the underlying cause.  Special logic is performed to ensure
     * that no JDK 1.4 VM is being used when the inner exception is displayed
     * (in order to prevent double printing).
     */
    public void printStackTrace( PrintStream ps )
    {
        this.source.printStackTrace( ps );
        if (shouldDisplayCause())
        {
            ps.println( getUnderlyingExceptionSeparator() );
            Throwable t = getCause();
            if (t == null)
            {
                ps.println( getUnknownExceptionString() );
            }
            else
            {
                t.printStackTrace( ps );
            }
        }
    }
    
    
    
    /**
     * For non-JDK 1.4 compatible VMs, this overrides the original behavior
     * to describe the underlying cause.  Special logic is performed to ensure
     * that no JDK 1.4 VM is being used when the inner exception is displayed
     * (in order to prevent double printing).
     */
    public void printStackTrace( PrintWriter pw )
    {
        this.source.printStackTrace( pw );
        if (shouldDisplayCause())
        {
            pw.println( getUnderlyingExceptionSeparator() );
            Throwable t = getCause();
            if (t == null)
            {
                pw.println( getUnknownExceptionString() );
            }
            else
            {
                t.printStackTrace( pw );
            }
        }
    }
    
    
    
    protected String getUnderlyingExceptionSeparator()
    {
        return "-------- Underlying exception --------";
    }
    
    
    protected String getUnknownExceptionString()
    {
        return "Unknown or non-existent exception";
    }
    
    
    protected boolean shouldDisplayCause()
    {
        // if the cause was never set, then don't display the cause.
        // if it was set (even if null), then we should display it.
        if (!this.causeSet)
        {
            return false;
        }
        
        // we need to do it.
        return true;
    }
}

