/*
 * Maildir folder access
 *
 * Copyright (C) 2004--2008  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <buffy/mailfolder/maildir.h>
#include <tests/test-utils.h>
#include <buffy/mailfolder/mailbox.h>

namespace tut {
using namespace buffy;
using namespace buffy::mailfolder;
using namespace wibble::tests;

struct buffy_mailfolder_maildir_shar {
	buffy_mailfolder_maildir_shar() {
	}
	~buffy_mailfolder_maildir_shar() {
	}
};
TESTGRP(buffy_mailfolder_maildir);

class MailFolderCounter : public MailFolderConsumer
{
	size_t m_count;
public:
	MailFolderCounter() : m_count(0) {}
	void consume(MailFolder& f) { ++m_count; }
	size_t count() const { return m_count; }
};

// Check an empty maildir
template<> template<>
void to::test<1>()
{
	MailFolder test(Maildir::accessFolder("maildir/empty"));
	ensure_equals(test.name(), "empty");
	ensure_equals(test.path(), "maildir/empty");
	ensure_equals(test.type(), "Maildir");

	ensure_equals(test.getMsgTotal(), -1);
	ensure_equals(test.getMsgUnread(), -1);
	ensure_equals(test.getMsgNew(), -1);
	ensure_equals(test.getMsgFlagged(), -1);
	ensure_equals(test.changed(), true);

	test.updateStatistics();
	ensure_equals(test.getMsgTotal(), 0);
	ensure_equals(test.getMsgUnread(), 0);
	ensure_equals(test.getMsgNew(), 0);
	ensure_equals(test.getMsgFlagged(), 0);
	ensure_equals(test.changed(), false);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/empty", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a non empty maildir
template<> template<>
void to::test<2>()
{
	MailFolder test(Maildir::accessFolder("maildir/test"));
	ensure_equals(test.name(), "test");
	ensure_equals(test.path(), "maildir/test");
	ensure_equals(test.type(), "Maildir");

	ensure_equals(test.getMsgTotal(), -1);
	ensure_equals(test.getMsgUnread(), -1);
	ensure_equals(test.getMsgNew(), -1);
	ensure_equals(test.getMsgFlagged(), -1);
	ensure_equals(test.changed(), true);

	test.updateStatistics();
	ensure_equals(test.getMsgTotal(), 3);
	ensure_equals(test.getMsgUnread(), 0);
	ensure_equals(test.getMsgNew(), 0);
	ensure_equals(test.getMsgFlagged(), 1);
	ensure_equals(test.changed(), false);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/test", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a maildir whose root is a broken symlink
template<> template<>
void to::test<3>()
{
	MailFolder test(Maildir::accessFolder("maildir/broken"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/broken", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir whose cur, new and tmp directories are broken symlinks
template<> template<>
void to::test<4>()
{
	MailFolder test(Maildir::accessFolder("maildir/broken1"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/broken1", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir whose root is a symlink pointing to a nonexisting directory
template<> template<>
void to::test<5>()
{
	MailFolder test(Maildir::accessFolder("maildir/loop"));
	ensure_equals((bool)test, false);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/loop", counter);
	ensure_equals(counter.count(), 0u);
}

// Check a maildir which has a submaildir that is a symlink pointing up
template<> template<>
void to::test<6>()
{
	MailFolder test(Maildir::accessFolder("maildir/loop1"));
	ensure_equals((bool)test, true);

	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir/loop1", counter);
	ensure_equals(counter.count(), 1u);
}

// Check a maildir which has a submaildir that is a symlink pointing up
template<> template<>
void to::test<7>()
{
	MailFolderCounter counter;
	Maildir::enumerateFolders("maildir", counter);
	ensure_equals(counter.count(), 4u);
}

}

// vim:set ts=4 sw=4:
