// Copyright ©2016 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package mathext

import (
	"math"
	"testing"
)

func TestGammaIncReg(t *testing.T) {
	t.Parallel()
	for i, test := range []struct {
		a, x, want float64
	}{
		// Results computed using scipy.special.gamminc
		{0, 0, 0},
		{0.0001, 1, 0.99997805936186279},
		{0.001, 0.005, 0.99528424172333985},
		{0.01, 10, 0.99999995718295021},
		{0.1, 10, 0.99999944520142825},
		{0.25, 0.75, 0.89993651328449831},
		{0.5, 0.5, 0.68268949213708596},
		{0.5, 2, 0.95449973610364147},
		{0.75, 2.5, 0.95053039734695643},
		{1, 0.5, 0.39346934028736652},
		{1, 1, 0.63212055882855778},
		{1.5, 0.75, 0.31772966966378746},
		{2.5, 1, 0.15085496391539038},
		{3, 0.05, 2.0067493624397931e-05},
		{3, 20, 0.99999954448504946},
		{5, 50, 1},
		{7, 10, 0.86985857911751696},
		{10, 0.9, 4.2519575433351128e-08},
		{10, 5, 0.031828057306204811},
		{25, 10, 4.6949381426799868e-05},
	} {
		if got := GammaIncReg(test.a, test.x); math.Abs(got-test.want) > 1e-10 {
			t.Errorf("test %d GammaIncReg(%g, %g) failed: got %g want %g", i, test.a, test.x, got, test.want)
		}
	}
}

func TestGammaIncRegComp(t *testing.T) {
	t.Parallel()
	for i, test := range []struct {
		a, x, want float64
	}{
		// Results computed using scipy.special.gammincc
		{0.00001, 0.075, 2.0866541002417804e-05},
		{0.0001, 1, 2.1940638138146658e-05},
		{0.001, 0.005, 0.0047157582766601536},
		{0.01, 0.9, 0.0026263432520514662},
		{0.25, 0.75, 0.10006348671550169},
		{0.5, 0.5, 0.31731050786291404},
		{0.75, 0.25, 0.65343980284081038},
		{0.9, 0.01, 0.98359881081593148},
		{1, 0, 1},
		{1, 0.075, 0.92774348632855297},
		{1, 1, 0.36787944117144233},
		{1, 10, 4.5399929762484861e-05},
		{1, math.Inf(1), 0},
		{3, 20, 4.5551495055892125e-07},
		{5, 10, 0.029252688076961127},
		{10, 3, 0.99889751186988451},
		{50, 25, 0.99999304669475242},
		{100, 10, 1},
		{500, 500, 0.49405285382921321},
		{500, 550, 0.014614408126291296},
	} {
		if got := GammaIncRegComp(test.a, test.x); math.Abs(got-test.want) > 1e-10 {
			t.Errorf("test %d GammaIncRegComp(%g, %g) failed: got %g want %g", i, test.a, test.x, got, test.want)
		}
	}
}

func TestGammaIncRegInv(t *testing.T) {
	t.Parallel()
	for i, test := range []struct {
		a, x, want float64
	}{
		// Results computed using scipy.special.gammincinv
		{0.001, 0.99, 2.4259428385570885e-05},
		{0.01, 0.99, 0.26505255025157959},
		{0.1, 0.5, 0.00059339110446022798},
		{0.2, 0.8, 0.26354363204872067},
		{0.25, 0.5, 0.043673802352873381},
		{0.5, 0.25, 0.050765522133810789},
		{0.5, 0.5, 0.22746821155978625},
		{0.75, 0.25, 0.15340752707472377},
		{1, 0, 0},
		{1, 0.075, 0.077961541469711862},
		{1, 1, math.Inf(1)},
		{2.5, 0.99, 7.5431362346944937},
		{10, 0.5, 9.6687146147141299},
		{25, 0.01, 14.853341349420646},
		{25, 0.99, 38.076945624506337},
		{50, 0.75, 54.570620535040511},
		{100, 0.25, 93.08583383712174},
		{1000, 0.01, 927.90815979664251},
		{1000, 0.99, 1075.0328320864389},
		{10000, 0.5, 9999.6666686420485},
	} {
		if got := GammaIncRegInv(test.a, test.x); math.Abs(got-test.want) > 1e-10 {
			t.Errorf("test %d GammaIncRegInv(%g, %g) failed: got %g want %g", i, test.a, test.x, got, test.want)
		}
	}
}

func TestGammaIncRegCompInv(t *testing.T) {
	t.Parallel()
	for i, test := range []struct {
		a, x, want float64
	}{
		// Results computed using scipy.special.gamminccinv
		{0.001, 0.01, 2.4259428385570885e-05},
		{0.01, 0.01, 0.26505255025158292},
		{0.03, 0.4, 2.316980536227699e-08},
		{0.1, 0.5, 0.00059339110446022798},
		{0.1, 0.75, 5.7917132949696076e-07},
		{0.25, 0.25, 0.26062600197823282},
		{0.5, 0.1, 1.3527717270477047},
		{0.5, 0.5, 0.22746821155978625},
		{0.75, 0.25, 1.0340914067758025},
		{1, 0, math.Inf(1)},
		{1, 0.5, 0.69314718055994529},
		{1, 1, 0},
		{3, 0.75, 1.727299417860519},
		{25, 0.4, 25.945791937289371},
		{25, 0.7, 22.156653488661991},
		{10, 0.5, 9.6687146147141299},
		{100, 0.25, 106.5510925269767},
		{1000, 0.01, 1075.0328320864389},
		{1000, 0.99, 927.90815979664251},
		{10000, 0.5, 9999.6666686420485},
	} {
		if got := GammaIncRegCompInv(test.a, test.x); math.Abs(got-test.want) > 1e-10 {
			t.Errorf("test %d GammaIncRegCompInv(%g, %g) failed: got %g want %g", i, test.a, test.x, got, test.want)
		}
	}
}
