-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-resultset.ads,v $
--  Description     : Ada Database Objects - Resultset object                --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/26 09:01:45 $
--  Version         : $Revision: 1.4 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  A result set is the collection of all records created by a query. The    --
--  contents of a result set is accessable by a cursor which defines the     --
--  the location of the current record in the result set.                    --
--                                                                           --
--                                                                           --
--                                                                           --
--                                                                           --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with GNU.DB.ADBC.Statement;                       use GNU.DB.ADBC.Statement;
with GNU.DB.ADBC.Row;                             use GNU.DB.ADBC.Row;
with GNU.DB.ADBC.Driver;

package GNU.DB.ADBC.Resultset is

   type Object( Stmt : Statement_ID ) is abstract tagged private;
   type Handle is access all Object'Class;

   Null_Handle : constant Handle := null;
   Null_ID     : constant Resultset_ID := Resultset_ID'First;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    allocate a resultset. This function is normally used by the drvier
   --#    in order to store his result set handle in the resultset manager.
   --# Preconditions  :
   --#    none
   --# Postconditions :
   --#
   --# Exceptions     :
   --#    Result_Table_Overflow - No id can be assigned to the handle
   --# Note           :
   ---------------------------------------------------------------------------
   function Allocate(
      Set : in Handle ) return Resultset_ID ;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Remove the identified handle from the result set table.
   --# Preconditions  :
   --#    none
   --# Postconditions :
   --#    P.1 - Resultset handle removed from table
   --# Exceptions     :
   --#    None
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Deallocate(
      Id  : in Resultset_ID );

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Return the object handle for the given resultset id.
   --# Preconditions  :
   --#    none
   --# Postconditions :
   --#    unchanged
   --# Exceptions     :
   --#    None
   --# Note           :
   ---------------------------------------------------------------------------
   function Get_Handle(
      Id  : in Resultset_ID ) return Handle;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Get the result set id of a given result set object
   --# Preconditions  :
   --#    none
   --# Postconditions :
   --#    unchanged
   --# Exceptions     :
   --#    None
   --# Note           :
   ---------------------------------------------------------------------------
   function ID(
      This : in Resultset.Object'Class ) return Resultset_ID;

   Resultset_Table_Overflow : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Readmode handling
   --# Preconditions  :
   --#    None
   --# Postconditions :
   --#
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   type Fetch_Mode_Type is (
         Previous,
         Next,
         First,
         Last
      );

   procedure Mode(
      This  : in Resultset_ID;
      Value : in Fetch_Mode_Type ) ;

   function Mode(
      This  : in Resultset_ID ) return Fetch_Mode_Type;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    get the number of records in the result set
   --# Preconditions  :
   --#    None
   --# Postconditions :
   --#
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Number_Of_Records(
      This  : in Resultset_ID;
      Value : in Natural );

   function Number_Of_Records(
      This  : in Resultset_ID ) return Natural;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    add an attribute for a result set
   --# Preconditions  :
   --#    R.1 - valid resultset
   --# Postconditions :
   --#
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   Max_Number_Of_Attributes : constant Positive := 50;

   procedure Attribute(
      This : in Resultset_ID;
      Name : in String );

   function Attribute(
      This : in Resultset_ID;
      Nbr  : in Positive ) return String;

   Attribute_Overflow     : exception;
   Attribute_Not_Existing : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Destroy the statement object
   --# Preconditions  :
   --#    R.1 - the object has been created
   --# Postconditions :
   --#    P.1 - The statement object is destroyed all references to it are
   --#          not valid any more.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
--     procedure Destroy(
--        This : in out Handle ) ;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Fetch next element from result set
   --# Preconditions  :
   --#    R.1 - handle is valid
   --# Postconditions :
   --#    P.1 - The fetch pointer is moved to next valid element
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Fetch(
      This : in Resultset_ID;
      Mode : in Fetch_Mode_Type := Next ) ;

   End_Of_Resultset_Exception   : exception;
   Begin_Of_Resultset_Exception : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    End_Of
   --# Preconditions  :
   --#    R.1 - handle is valid
   --# Postconditions :
   --#    P.1 - False if there is s record left in the result set.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   function End_Of_Result(
      This : in Resultset_ID ) return Boolean;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Begin_Of_Result
   --# Preconditions  :
   --#    R.1 - handle is valid
   --# Postconditions :
   --#    P.1 - False if we have reached the begin of the result set
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   function Begin_Of_result(
      This : in Resultset_ID ) return Boolean;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Domain accessor
   --# Preconditions  :
   --#    R.1 -
   --# Postconditions :
   --#    P.1 -
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------

   -------------------
   -- String_Domain --
   -------------------
   generic
      Result : Resultset_ID;                 -- resultset id
      Name   : String;                       -- fieldname
      Size   : Positive;
   package String_Domain is

      function Is_Null return Boolean;
      function  Value(
         Trim : Boolean := False  )  return String;

      procedure Set( Value : in String );

   end String_Domain;

   --------------------
   -- Numeric_Domain --
   --------------------
   generic
      Result : Resultset_ID;                 -- resultset id
      Name   : String;                       -- fieldname
      type Number_Type is range <>;          -- Number type
   package Integer_Domain is

      function Is_Null return Boolean;
      function  Value return Number_Type;
      procedure Set( Value : in Number_Type  );

   end Integer_Domain;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Domain accessor
   --# Preconditions  :
   --#    R.1 -
   --# Postconditions :
   --#    P.1 -
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Fetch(
      This   : in out Object;
      Result : out Row.Handle;
      Mode   : in Fetch_Mode_Type := Next ) is abstract;

   function Driver_Handle(
      This  : in Resultset_ID ) return Driver.Handle ;

--     procedure Driver_Handle(
--        This  : in Resultset_ID;
--        Db    : in Driver.Handle );

   ---=====================================================================---
private
   type Object_Data;
   type Object_Data_Access is access Object_Data;

   type Object( Stmt : Statement_ID ) is abstract tagged record
         Data : Object_Data_Access := null;
      end record;

end GNU.DB.ADBC.Resultset;

