static char rcsid[] = "$Id: 47fa1b5d9a1fa3f628d011f3cb2f1c7c069816e4 $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif
#ifndef HAVE_MEMMOVE
#define memmove(d,s,n) bcopy((s),(d),(n))
#endif

#include "stage1hr.h"
#include "stage1hr-single.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>		/* For rint */
#include <string.h>		/* For memset */

#include "assert.h"
#include "mem.h"
#include "types.h"		/* Needed for HAVE_64_BIT */
#include "univcoord.h"

#include "list.h"
#include "compress.h"
#include "record.h"

#include "genomebits_mismatches.h" /* For MISMATCH_EXTRA */
#include "orderstat.h"
#include "transcriptome-search.h"
#include "tr-extension-search.h"
#include "kmer-search.h"
#include "extension-search.h"
#include "merge-search.h"
#include "segment-search.h"

#include "trpath-solve.h"
#include "trpath-convert.h"
#include "path-solve.h"
#include "path-fusion.h"

#include "transcript-remap.h"
#include "transcript-velocity.h"
#include "path-eval.h"


#define MIN_SIZELIMIT 100
#define MAX_HITS_EXACT 100	/* Excessive exact paths indicate a repetitive sequence */
#define MAX_END_POSITIONS 100	/* For END method.  Could be different for DNA-seq compared with RNA-seq */


static Mode_T mode;
static int index1part;
static int index1interval;
static int index1part_tr;

static Transcriptome_T transcriptome;
static bool transcriptome_align_p;
static bool genome_align_p;

static double user_nmismatches_filter_float;
static double user_mincoverage_filter_float;

static double max_middle_insertions_float;
static double max_middle_deletions_float;

static bool splicingp;
static Chrpos_T shortsplicedist;
static Chrpos_T shortsplicedist_novelend;



#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif



#define T Stage1_T

static int
determine_sizelimit (T this, int querylength) {
  int cutoff, *set, count;
  int n;
  int query_lastpos, querypos;

  assert(querylength >= index1part);

  query_lastpos = querylength - index1part;
  set = (int *) MALLOC(2*(query_lastpos+1)*sizeof(int));
  n = 0;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    if (this->validp[querypos] == true) {
      set[n++] = count = this->plus_npositions[querypos];
      set[n++] = count = this->minus_npositions[querypos];
    }
  }

  if (n < 5) {
    cutoff = MIN_SIZELIMIT;
  } else if ((cutoff = Orderstat_int_pct_inplace(set,n,/*pct*/0.60)) < MIN_SIZELIMIT) {
    cutoff = MIN_SIZELIMIT;
  }
  FREE(set);

  return cutoff;
}


#ifdef DEBUG
static void
list_paths (List_T list, char *destination, bool expected_sensedir) {
  List_T p;
  Path_T path;

  for (p = list; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);
    printf("Destination %s: ",destination);
    Path_print(path);
    assert(path->sensedir == expected_sensedir);
  }

  return;
}
#endif


/* Benchmarks show slight speed improvement with TR_EXACT2 before TR_ANYPAIR */
Method_T
single_read_next_method_trdiagonal (Method_T last_method, T this, int querylength) {

  if (last_method < TR_EXACT1) {
    /* 1.1.  TR_EXACT1 */
    debug(printf("Read: 1.1.  Running Transcriptome_exact1\n"));
    Stage1_init_end_oligos_tr(this,querylength);
    Stage1_init_end_positions_tr(this,querylength);
    Transcriptome_exact1(&this->sense_trnums,&this->sense_troffsets,&this->sense_trhighs,
			 &this->sense_trdiagonals,&this->n_sense_trdiagonals,
			 &this->antisense_trnums,&this->antisense_troffsets,&this->antisense_trhighs,
			 &this->antisense_trdiagonals,&this->n_antisense_trdiagonals,
			 this,querylength);
    return TR_EXACT1;

  } else if (last_method < TR_EXACT2) {
    Stage1_trdiagonals_gc(this);
    
    Stage1_fill_all_oligos_tr(this,querylength);
    Stage1_init_end2_positions_tr(this,querylength);
    Transcriptome_exact2(&this->sense_trnums,&this->sense_troffsets,&this->sense_trhighs,
			 &this->sense_trdiagonals,&this->n_sense_trdiagonals,
			 &this->antisense_trnums,&this->antisense_troffsets,&this->antisense_trhighs,
			 &this->antisense_trdiagonals,&this->n_antisense_trdiagonals,
			 this,querylength);
    return TR_EXACT2;

  } else if (last_method < TR_ANYPAIR) {
    Stage1_trdiagonals_gc(this);

    Stage1_fill_all_positions_tr(this,querylength);
    Transcriptome_anypair(&this->sense_trnums,&this->sense_troffsets,&this->sense_trhighs,
			  &this->sense_trdiagonals,&this->sense_tstarts,&this->sense_tends,
			  &this->n_sense_trdiagonals,
			  &this->antisense_trnums,&this->antisense_troffsets,&this->antisense_trhighs,
			  &this->antisense_trdiagonals,&this->antisense_tstarts,&this->antisense_tends,
			  &this->n_antisense_trdiagonals,this,querylength);
    return TR_ANYPAIR;

  } else {
    fprintf(stderr,"No trdiagonal method after TR_ANYPAIR\n");
    abort();
  }
}


Method_T
single_read_next_method_tr (int *found_score, Method_T last_method,

			    List_T *sense_trpaths, List_T *antisense_trpaths,

			    T this, int genestrand, int querylength,
			    int *mismatch_positions_alloc,
			    Compress_T query_compress_fwd, Compress_T query_compress_rev,

			    int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
			 
			    Trdiagpool_T trdiagpool,
			    Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			    Listpool_T listpool, Trpathpool_T trpathpool, Pathpool_T pathpool,
			    Hitlistpool_T hitlistpool, bool appendp) {

  if (appendp == false) {
    *sense_trpaths = *antisense_trpaths = (List_T) NULL;
  }

  if (last_method < TR_EXT) {

#if 0
    /* Supplanted by new Tr_extension_search,which also uses
       Genomebits_trim procedures, but still offers some speed
       advantage */
    /* 1.2.  TR_END */
    debug(printf("Read: 1.2.  Running Transcriptome_search_end\n"));
    Transcriptome_search_end(&(*found_score),&(*sense_trpaths),&(*antisense_trpaths),

			     this,querylength,query_compress_fwd,query_compress_rev, 
			     intlistpool,uintlistpool,trpathpool,hitlistpool,
			     /*method*/TR_END);
    return TR_END;

  } else if (last_method == TR_END) {
#endif

    /* 1.3.  TR_EXT */
    debug(printf("Read: 1.3.  Running Tr_extension_search\n"));
    Stage1_fill_all_oligos_tr(this,querylength);
    Tr_extension_search(&(*found_score),&(*sense_trpaths),&(*antisense_trpaths),

			this,querylength,mismatch_positions_alloc,
			query_compress_fwd,query_compress_rev, 

			trdiagpool,intlistpool,uintlistpool,listpool,
			trpathpool,pathpool,hitlistpool,
			max_insertionlen,max_deletionlen,nmismatches_allowed,
			genestrand,/*method*/TR_EXT);
    return TR_EXT;
    
  } else {
    fprintf(stderr,"No method after TR_EXT\n");
    abort();
  }
}


/* Faster to have KMER_ANYPAIR before KMER_EXACT2 */
Method_T
single_read_next_method_univdiagonal (Method_T last_method,
				      Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
				      Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,
				      T this, T mate, int genestrand, int querylength,
				      Chrpos_T overall_max_distance, bool first_read_p) {

  *univdiagonals_gplus = *univdiagonals_gminus = (Univcoord_T *) NULL;
  *nunivdiagonals_gplus = *nunivdiagonals_gminus = 0;

  if (last_method < KMER_EXACT1) {
    debug(printf("Read: 1.  Running Kmer_exact1\n"));
    Stage1_init_end_gen(this,querylength,genestrand);
    Kmer_exact1(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		&(*univdiagonals_gminus),&(*nunivdiagonals_gminus),
		this,querylength);
    return KMER_EXACT1;

  } else if (last_method < KMER_ANYPAIR) {
    /* Anypair needs to know npositions for all querypos */
    Stage1_fill_all_oligos_gen(this,querylength,genestrand);
    Stage1_fill_all_positions_gen(this,querylength,genestrand);
    Kmer_anypair(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		 &(*univdiagonals_gminus),&(*nunivdiagonals_gminus),
		 this,querylength);
    return KMER_ANYPAIR;

  } else if (last_method < KMER_EXACT2) {
    debug(printf("Read: 1.  Running Kmer_exact2\n"));
    assert(this->all_oligos_gen_filledp == true);
    assert(this->all_positions_gen_filledp == true);
    Kmer_exact2(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		&(*univdiagonals_gminus),&(*nunivdiagonals_gminus),
		this,querylength);
    return KMER_EXACT2;

  } else if (last_method < KMER_PREVALENT) {
    /* Slower methods, requiring merging all positions */
    assert(this->all_oligos_gen_filledp == true);
    assert(this->all_positions_gen_filledp == true);
    Kmer_prevalent(&this->all_univdiagonals_gplus,&this->all_nunivdiagonals_gplus,
		   &this->max_npositions_gplus,
		   &this->all_univdiagonals_gminus,&this->all_nunivdiagonals_gminus,
		   &this->max_npositions_gminus,
		   &(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		   &(*univdiagonals_gminus),&(*nunivdiagonals_gminus),
		   this,querylength);
    return KMER_PREVALENT;

  } else if (mate != NULL && last_method < KMER_ANCHORED) {
    if (first_read_p == true) {
      Kmer_anchored_5(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		      &(*univdiagonals_gminus),&(*nunivdiagonals_gminus),

		      this->all_univdiagonals_gplus,this->all_nunivdiagonals_gplus,
		      this->all_univdiagonals_gminus,this->all_nunivdiagonals_gminus,

		      mate->univdiagonals_gplus,mate->nunivdiagonals_gplus,
		      mate->univdiagonals_gminus,mate->nunivdiagonals_gminus,

		      overall_max_distance);
    } else {
      Kmer_anchored_3(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		      &(*univdiagonals_gminus),&(*nunivdiagonals_gminus),

		      mate->univdiagonals_gplus,mate->nunivdiagonals_gplus,
		      mate->univdiagonals_gminus,mate->nunivdiagonals_gminus,

		      this->all_univdiagonals_gplus,this->all_nunivdiagonals_gplus,
		      this->all_univdiagonals_gminus,this->all_nunivdiagonals_gminus,

		      overall_max_distance);
    }

    return KMER_ANCHORED;

  } else if (last_method < KMER_WIDEST) {
    Kmer_widest(&(*univdiagonals_gplus),&(*nunivdiagonals_gplus),
		&(*univdiagonals_gminus),&(*nunivdiagonals_gminus),

		this->all_univdiagonals_gplus,this->all_nunivdiagonals_gplus,
		this->max_npositions_gplus,
		this->all_univdiagonals_gminus,this->all_nunivdiagonals_gminus,
		this->max_npositions_gminus,
		this,querylength);
    return KMER_WIDEST;

  } else {
    fprintf(stderr,"No univdiagonal method after KMER_WIDEST\n");
    abort();
  }
}


Method_T
single_read_next_method_gen (int *found_score, Method_T last_method,

			     List_T *sense_paths_gplus, List_T *sense_paths_gminus, 
			     List_T *antisense_paths_gplus, List_T *antisense_paths_gminus, 

			     T this, EF64_T repetitive_ef64, int genestrand,

			     Shortread_T queryseq, char *queryuc_ptr, char *queryrc, int querylength,
			     Knownsplicing_T knownsplicing, Knownindels_T knownindels,
			     int *mismatch_positions_alloc, Univcoord_T *novel_diagonals_alloc, unsigned short *localdb_alloc,
			     Compress_T query_compress_fwd, Compress_T query_compress_rev,

			     int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
			     Chrpos_T overall_max_distance, Chrpos_T overall_end_distance,
			 
			     Univdiagpool_T univdiagpool,
			     Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			     Univcoordlistpool_T univcoordlistpool, Listpool_T listpool, 
			     Pathpool_T pathpool, Transcriptpool_T transcriptpool,
			     Vectorpool_T vectorpool, Hitlistpool_T hitlistpool, Spliceendsgen_T spliceendsgen,
			     bool paired_end_p, bool first_read_p, bool appendp) {

  /* For segment_search */
  int sizelimit;
  struct Record_T *plus_records = NULL, *minus_records = NULL;
  int plus_nrecords = 0, minus_nrecords = 0;


  if (appendp == false) {
    *sense_paths_gplus = *sense_paths_gminus =
      *antisense_paths_gplus = *antisense_paths_gminus = (List_T) NULL;
  }

  if (last_method < EXT) {
    if (this->all_oligos_gen_filledp == false) {
      Stage1_fill_all_oligos_gen(this,querylength,genestrand);
    }
    Extension_search(&(*found_score),
		     
		     &this->unextended_sense_paths_gplus,&this->unextended_sense_paths_gminus,
		     &this->unextended_antisense_paths_gplus,&this->unextended_antisense_paths_gminus,
		     
		     &(*sense_paths_gplus),&(*sense_paths_gminus),
		     &(*antisense_paths_gplus),&(*antisense_paths_gminus),
		     
		     this,queryseq,queryuc_ptr,queryrc,querylength,knownsplicing,knownindels,
		     mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
		     query_compress_fwd,query_compress_rev,
		     univdiagpool,intlistpool,uintlistpool,univcoordlistpool,listpool,
		     pathpool,transcriptpool,vectorpool,hitlistpool,spliceendsgen,
		     max_insertionlen,max_deletionlen,nmismatches_allowed,overall_end_distance,
		     genestrand,paired_end_p,first_read_p,/*method*/EXT,
		     /*find_splices_p*/true);
    return EXT;

  } else if (last_method < SEGMENT1) {
    if (this->all_oligos_gen_filledp == false) {
      Stage1_fill_all_oligos_gen(this,querylength,genestrand);
    }
    if (this->all_positions_gen_filledp == false) {
      Stage1_fill_all_positions_gen(this,querylength,genestrand);
    }

    /* Need sizelimit to constrain segment search */
    sizelimit = determine_sizelimit(this,querylength);
    
    debug(printf("Starting Segment_identify on plus strand\n"));
    plus_records = Segment_identify(&plus_nrecords,
#ifdef LARGE_GENOMES
				    this->plus_positions_high,
#endif
				    this->plus_positions,this->plus_npositions,this->validp,
				    this->forward_oligos,repetitive_ef64,
#ifdef LARGE_GENOMES
				    this->stream_high_alloc,this->stream_low_alloc,
#else
				    this->streamptr_alloc,
#endif
				    this->streamsize_alloc,this->querypos_diagterm_alloc,this->mergeinfo,
				    overall_max_distance,querylength,sizelimit,/*plusp*/true);
    debug(printf("Done\n"));
    
    debug(printf("Starting Segment_identify on minus strand\n"));
    minus_records = Segment_identify(&minus_nrecords,
#ifdef LARGE_GENOMES
				     this->minus_positions_high,
#endif
				     this->minus_positions,this->minus_npositions,this->validp,
				     this->revcomp_oligos,repetitive_ef64,
#ifdef LARGE_GENOMES
				     this->stream_high_alloc,this->stream_low_alloc,
#else
				     this->streamptr_alloc,
#endif
				     this->streamsize_alloc,this->querypos_diagterm_alloc,this->mergeinfo,
				     overall_max_distance,querylength,sizelimit,/*plusp*/false);
    debug(printf("Done\n"));
    
    debug(printf("Read %s: Running Segment_search_all\n",first_read_p ? "5'" : "3'"));
    Segment_search_all(&(*found_score),

		       &this->unextended_sense_paths_gplus,&this->unextended_sense_paths_gminus,
		       &this->unextended_antisense_paths_gplus,&this->unextended_antisense_paths_gminus,
		       &(*sense_paths_gplus),&(*sense_paths_gminus),
		       &(*antisense_paths_gplus),&(*antisense_paths_gminus),
		       
		       plus_records,plus_nrecords,minus_records,minus_nrecords,
		       queryseq,queryuc_ptr,queryrc,querylength,
		       mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
		       this,knownsplicing,knownindels,
		       query_compress_fwd,query_compress_rev,
		       max_insertionlen,max_deletionlen,nmismatches_allowed,
		       overall_max_distance,overall_end_distance,
		       genestrand,paired_end_p,first_read_p,
		       univdiagpool,intlistpool,uintlistpool,univcoordlistpool,listpool,
		       pathpool,transcriptpool,vectorpool,hitlistpool,
		       spliceendsgen,/*method*/SEGMENT1,/*find_splices_p*/true);
    debug(printf("Done\n"));
    
    FREE(minus_records);
    FREE(plus_records);

    return SEGMENT1;

#if 0
  } else if (last_method < MERGE) {
    Merge_search(&(*found_score),

		 &this->unextended_sense_paths_gplus,&this->unextended_sense_paths_gminus,
		 &this->unextended_antisense_paths_gplus,&this->unextended_antisense_paths_gminus,
		 &(*sense_paths_gplus),&(*sense_paths_gminus),
		 &(*antisense_paths_gplus),&(*antisense_paths_gminus),

		 this,knownsplicing,querylength,query_compress_fwd,query_compress_rev,
		 max_insertionlen,max_deletionlen,overall_end_distance,
		 univdiagpool,intlistpool,univcoordlistpool,listpool,
		 pathpool,vectorpool,hitlistpool,/*method*/MERGE);
    return MERGE;
#endif

  } else {
    fprintf(stderr,"No method after SEGMENT1\n");
    abort();
  }
}



#if 0
/* Slow method */
static void
single_read_search_complete (int *found_score,

			     List_T *sense_paths_gplus, List_T *sense_paths_gminus, 
			     List_T *antisense_paths_gplus, List_T *antisense_paths_gminus, 

			     T this, int genestrand,

			     Shortread_T queryseq, char *queryuc_ptr, char *queryrc, int querylength,
			     Knownsplicing_T knownsplicing, Knownindels_T knownindels,
			     int *mismatch_positions_alloc, Univcoord_T *novel_diagonals_alloc, unsigned short *localdb_alloc,
			     Compress_T query_compress_fwd, Compress_T query_compress_rev,

			     int max_insertionlen, int max_deletionlen,
			 
			     Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			     Univcoordlistpool_T univcoordlistpool, Listpool_T listpool, 
			     Pathpool_T pathpool, Transcriptpool_T transcriptpool,
			     Vectorpool_T vectorpool, Hitlistpool_T hitlistpool, Spliceendsgen_T spliceendsgen,
			     bool paired_end_p, bool first_read_p, Pass_T pass) {

  assert(this->all_positions_gen_filledp == false);
  Stage1_fill_all_positions_gen(this,querylength,genestrand);

  Kmer_search_complete(&(*found_score),
		       &this->unextended_sense_paths_gplus,&this->unextended_sense_paths_gminus,
		       &this->unextended_antisense_paths_gplus,&this->unextended_antisense_paths_gminus,

		       &(*sense_paths_gplus),&(*sense_paths_gminus),
		       &(*antisense_paths_gplus),&(*antisense_paths_gminus),
			 
		       this,queryseq,queryuc_ptr,queryrc,querylength,genestrand,
			 
		       this->streamspace_max_alloc,this->streamspace_alloc,
		       this->streamptr_alloc,this->streamsize_alloc, 
			 
		       this->mergeinfo,this->indelinfo,this->spliceinfo,
		       knownsplicing,knownindels,
			 
#ifdef LARGE_GENOMES
		       this->gplus_stream_high_array_5,this->gplus_stream_low_array_5,
		       this->gminus_stream_high_array_5,this->gminus_stream_low_array_5,
#else
		       this->gplus_stream_array_5,this->gminus_stream_array_5,
#endif
		       this->gplus_streamsize_array_5,this->gplus_diagterm_array_5,
		       this->gminus_streamsize_array_5,this->gminus_diagterm_array_5,

		       mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
			 
		       query_compress_fwd,query_compress_rev,max_insertionlen,max_deletionlen,
		       intlistpool,uintlistpool,univcoordlistpool,listpool,pathpool,
		       transcriptpool,vectorpool,hitlistpool,
		       spliceendsgen,paired_end_p,first_read_p,/*method*/KMER_COMPLETE,pass);

#if 0
  if (transcriptome_align_p == true) {
    *sense_paths_gplus = Path_consolidate(*sense_paths_gplus,queryseq,
					  query_compress_fwd,query_compress_rev,
					  uintlistpool,intlistpool,univcoordlistpool,
					  listpool,pathpool,transcriptpool,hitlistpool);
    *sense_paths_gminus = Path_consolidate(*sense_paths_gminus,queryseq,
					   query_compress_fwd,query_compress_rev,
					   uintlistpool,intlistpool,univcoordlistpool,
					   listpool,pathpool,transcriptpool,hitlistpool);
    *antisense_paths_gplus = Path_consolidate(*antisense_paths_gplus,queryseq,
					      query_compress_fwd,query_compress_rev,
					      uintlistpool,intlistpool,univcoordlistpool,
					      listpool,pathpool,transcriptpool,hitlistpool);
    *antisense_paths_gminus = Path_consolidate(*antisense_paths_gminus,queryseq,
					       query_compress_fwd,query_compress_rev,
					       uintlistpool,intlistpool,univcoordlistpool,
					       listpool,pathpool,transcriptpool,hitlistpool);
  }
#endif

#ifdef DEBUG
  list_paths(*sense_paths_gplus,"sense_paths_gplus",/*expected_sensedir*/SENSE_FORWARD);
  list_paths(*sense_paths_gminus,"sense_paths_gminus",/*expected_sensedir*/SENSE_FORWARD);
  list_paths(*antisense_paths_gplus,"antisense_paths_gplus",/*expected_sensedir*/SENSE_ANTI);
  list_paths(*antisense_paths_gminus,"antisense_paths_gminus",/*expected_sensedir*/SENSE_ANTI);
#endif

  return;
}
#endif



static List_T
solve_unsolved_paths (int *found_score, List_T solved_paths, List_T *unsolved_paths,
		      int sensedir, int genestrand,
		      int max_insertionlen, int max_deletionlen, Compress_T query_compress,
		      Compress_T query_compress_fwd, Compress_T query_compress_rev,
		      Shortread_T queryseq, int querylength,
		      Stage1_T this, Knownsplicing_T knownsplicing,
		      Uintlistpool_T uintlistpool, Intlistpool_T intlistpool,
		      Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
		      Pathpool_T pathpool, Transcriptpool_T transcriptpool, Hitlistpool_T hitlistpool) {
  Path_T newpath, path;
  List_T p;

  for (p = *unsolved_paths; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);

    /* Path_solve_junctions returns the same path */
    if ((newpath = Path_solve_junctions(&(*found_score),path,sensedir,genestrand,
					max_insertionlen,max_deletionlen,
					query_compress,query_compress_fwd,query_compress_rev,
					queryseq,querylength,this,knownsplicing,
					uintlistpool,intlistpool,univcoordlistpool,listpool,
					pathpool,transcriptpool)) == NULL) {
      Path_free(&path,intlistpool,univcoordlistpool,
		listpool,pathpool,transcriptpool,hitlistpool);

    } else {
      newpath->completep = true;
      solved_paths = Hitlist_push(solved_paths,hitlistpool,(void *) newpath
				  hitlistpool_trace(__FILE__,__LINE__));
    }
  }

  Hitlistpool_free_list(&(*unsolved_paths),hitlistpool
			hitlistpool_trace(__FILE__,__LINE__));

  return solved_paths;
}


void
single_read_unsolved_tr (int *found_score,

			 List_T *sense_paths_gplus, List_T *sense_paths_gminus, 
			 List_T *antisense_paths_gplus, List_T *antisense_paths_gminus, 

			 T this, Shortread_T queryseq, int querylength, Knownsplicing_T knownsplicing,
			 Compress_T query_compress_fwd, Compress_T query_compress_rev,

			 int max_insertionlen, int max_deletionlen,
			 
			 Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			 Univcoordlistpool_T univcoordlistpool, Listpool_T listpool, 
			 Pathpool_T pathpool, Transcriptpool_T transcriptpool,
			 Hitlistpool_T hitlistpool) {

  *sense_paths_gplus = solve_unsolved_paths(&(*found_score),*sense_paths_gplus,
					    &this->unsolved_sense_paths_gplus,
					    /*sensedir*/SENSE_FORWARD,/*genestrand*/0,
					    max_insertionlen,max_deletionlen,
					    /*query_compress*/query_compress_fwd,
					    query_compress_fwd,query_compress_rev,queryseq,querylength,
					    this,knownsplicing,
					    uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
					    transcriptpool,hitlistpool);

  *sense_paths_gminus = solve_unsolved_paths(&(*found_score),*sense_paths_gminus,
					     &this->unsolved_sense_paths_gminus,
					     /*sensedir*/SENSE_FORWARD,/*genestrand*/0,
					     max_insertionlen,max_deletionlen,
					     /*query_compress*/query_compress_rev,
					     query_compress_fwd,query_compress_rev,queryseq,querylength,
					     this,knownsplicing,
					     uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
					     transcriptpool,hitlistpool);

  *antisense_paths_gplus = solve_unsolved_paths(&(*found_score),*antisense_paths_gplus,
						&this->unsolved_antisense_paths_gplus,
						/*sensedir*/SENSE_ANTI,/*genestrand*/0,
						max_insertionlen,max_deletionlen,
						/*query_compress*/query_compress_fwd,
						query_compress_fwd,query_compress_rev,queryseq,querylength,
						this,knownsplicing,
						uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
						transcriptpool,hitlistpool);
  
  *antisense_paths_gminus = solve_unsolved_paths(&(*found_score),*antisense_paths_gminus,
						 &this->unsolved_antisense_paths_gminus,
						 /*sensedir*/SENSE_ANTI,/*genestrand*/0,
						 max_insertionlen,max_deletionlen,
						 /*query_compress*/query_compress_rev,
						 query_compress_fwd,query_compress_rev,queryseq,querylength,
						 this,knownsplicing,
						 uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
						 transcriptpool,hitlistpool);

  *sense_paths_gplus = Path_consolidate(*sense_paths_gplus,queryseq,
					query_compress_fwd,query_compress_rev,
					uintlistpool,intlistpool,univcoordlistpool,
					listpool,pathpool,transcriptpool,hitlistpool);
  *sense_paths_gminus = Path_consolidate(*sense_paths_gminus,queryseq,
					 query_compress_fwd,query_compress_rev,
					 uintlistpool,intlistpool,univcoordlistpool,
					 listpool,pathpool,transcriptpool,hitlistpool);
  *antisense_paths_gplus = Path_consolidate(*antisense_paths_gplus,queryseq,
					    query_compress_fwd,query_compress_rev,
					    uintlistpool,intlistpool,univcoordlistpool,
					    listpool,pathpool,transcriptpool,hitlistpool);
  *antisense_paths_gminus = Path_consolidate(*antisense_paths_gminus,queryseq,
					     query_compress_fwd,query_compress_rev,
					     uintlistpool,intlistpool,univcoordlistpool,
					     listpool,pathpool,transcriptpool,hitlistpool);

#ifdef DEBUG
  list_paths(*sense_paths_gplus,"sense_paths_gplus",/*expected_sensedir*/SENSE_FORWARD);
  list_paths(*sense_paths_gminus,"sense_paths_gminus",/*expected_sensedir*/SENSE_FORWARD);
  list_paths(*antisense_paths_gplus,"antisense_paths_gplus",/*expected_sensedir*/SENSE_ANTI);
  list_paths(*antisense_paths_gminus,"antisense_paths_gminus",/*expected_sensedir*/SENSE_ANTI);
#endif

  return;
}


/* Populates extended fields in Stage1_T object */
static void
single_read_extend (int *found_score, T this,

		    List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		    List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		    Shortread_T queryseq, char *queryuc_ptr, char *queryrc, int querylength,
		    Knownsplicing_T knownsplicing, Knownindels_T knownindels,
		    int *mismatch_positions_alloc, Univcoord_T *novel_diagonals_alloc, unsigned short *localdb_alloc,
		    Compress_T query_compress_fwd, Compress_T query_compress_rev,

		    int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
		    Chrpos_T overall_end_distance,

		    Intlistpool_T intlistpool,
		    Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
		    Listpool_T listpool, Pathpool_T pathpool, Transcriptpool_T transcriptpool,
		    Vectorpool_T vectorpool, Hitlistpool_T hitlistpool, Spliceendsgen_T spliceendsgen) {

  List_T newpaths, p;
  bool completep;

  /* Attempt to extend paths */
#if 0
  /* Path_filter appears to select incorrect splice lengths */
  debug(printf("Before Path_filter: %d unextended_sense_paths_gplus\n",
	       List_length(this->unextended_sense_paths_gplus)));
  debug(printf("Before Path_filter: %d unextended_antisense_paths_gplus\n",
	       List_length(this->unextended_antisense_paths_gplus)));
  debug(printf("Before Path_filter: %d unextended_sense_paths_gminus\n",
	       List_length(this->unextended_sense_paths_gminus)));
  debug(printf("Before Path_filter: %d unextended_antisense_paths_gminus\n",
	       List_length(this->unextended_antisense_paths_gminus)));

  this->unextended_sense_paths_gplus = Path_filter(this->unextended_sense_paths_gplus,
						   intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_sense_paths_gminus = Path_filter(this->unextended_sense_paths_gminus,
						    intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_antisense_paths_gplus = Path_filter(this->unextended_antisense_paths_gplus,
						       intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_antisense_paths_gminus = Path_filter(this->unextended_antisense_paths_gminus,
							intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
#else
  this->unextended_sense_paths_gplus = Path_unique(this->unextended_sense_paths_gplus,
						   intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_sense_paths_gminus = Path_unique(this->unextended_sense_paths_gminus,
						    intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_antisense_paths_gplus = Path_unique(this->unextended_antisense_paths_gplus,
						       intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
  this->unextended_antisense_paths_gminus = Path_unique(this->unextended_antisense_paths_gminus,
							intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,hitlistpool);
#endif
      
  debug(printf("Have %d unextended_sense_paths_gplus\n",List_length(this->unextended_sense_paths_gplus)));
  for (p = this->unextended_sense_paths_gplus; p != NULL; p = List_next(p)) {
    newpaths = Path_extend(&completep,&(*found_score),
			   /*original_path*/(Path_T) List_head(p),
			   queryseq,/*queryptr*/queryuc_ptr,querylength,
			   mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
			   this,knownsplicing,knownindels,
			   /*query_compress*/query_compress_fwd,
			   query_compress_fwd,query_compress_rev,/*genestrand*/0,
			   max_insertionlen,max_deletionlen,overall_end_distance,
			   nmismatches_allowed,/*paired_end_p*/false,/*lowp*/true,
			   intlistpool,uintlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,
			   vectorpool,hitlistpool,spliceendsgen,
			   /*extend_qstart_p*/true,/*extend_qend_p*/true);
    if (completep == true) {
      debug(printf("Found extended_sense_paths_gplus\n"));
      *sense_paths_gplus = List_append(newpaths,*sense_paths_gplus);
    } else {
      this->unextended_sense_paths_gplus = List_append(newpaths,this->unextended_sense_paths_gplus);
    }
  }

  debug(printf("Have %d unextended_sense_paths_gminus\n",List_length(this->unextended_sense_paths_gminus)));
  for (p = this->unextended_sense_paths_gminus; p != NULL; p = List_next(p)) {
    newpaths = Path_extend(&completep,&(*found_score),/*original_path*/(Path_T) List_head(p),
			   queryseq,/*queryptr*/queryrc,querylength,
			   mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
			   this,knownsplicing,knownindels,
			   /*query_compress*/query_compress_rev,
			   query_compress_fwd,query_compress_rev,/*genestrand*/0,
			   max_insertionlen,max_deletionlen,overall_end_distance,
			   nmismatches_allowed,/*paired_end_p*/false,/*lowp*/true,
			   intlistpool,uintlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,
			   vectorpool,hitlistpool,spliceendsgen,
			   /*extend_qstart_p*/true,/*extend_qend_p*/true);
    if (completep == true) {
      debug(printf("Found extended_sense_paths_gminus\n"));
      *sense_paths_gminus = List_append(newpaths,*sense_paths_gminus);
    } else {
      this->unextended_sense_paths_gminus = List_append(newpaths,this->unextended_sense_paths_gminus);
    }
  }

  debug(printf("Have %d unextended_antisense_paths_gplus\n",List_length(this->unextended_antisense_paths_gplus)));
  for (p = this->unextended_antisense_paths_gplus; p != NULL; p = List_next(p)) {
    newpaths = Path_extend(&completep,&(*found_score),/*original_path*/(Path_T) List_head(p),
			   queryseq,/*queryptr*/queryuc_ptr,querylength,
			   mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
			   this,knownsplicing,knownindels,
			   /*query_compress*/query_compress_fwd,
			   query_compress_fwd,query_compress_rev,/*genestrand*/0,
			   max_insertionlen,max_deletionlen,overall_end_distance,
			   nmismatches_allowed,/*paired_end_p*/false,/*lowp*/true,
			   intlistpool,uintlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,
			   vectorpool,hitlistpool,spliceendsgen,
			   /*extend_qstart_p*/true,/*extend_qend_p*/true);
    if (completep == true) {
      debug(printf("Found extended_antisense_paths_gplus\n"));
      *antisense_paths_gplus = List_append(newpaths,*antisense_paths_gplus);
    } else {
      this->unextended_antisense_paths_gplus = List_append(newpaths,this->unextended_antisense_paths_gplus);
    }
  }

  debug(printf("Have %d unextended_antisense_paths_gminus\n",List_length(this->unextended_antisense_paths_gminus)));
  for (p = this->unextended_antisense_paths_gminus; p != NULL; p = List_next(p)) {
    newpaths = Path_extend(&completep,&(*found_score),/*original_path*/(Path_T) List_head(p),
			   queryseq,/*queryptr*/queryrc,querylength,
			   mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
			   this,knownsplicing,knownindels,
			   /*query_compress*/query_compress_rev,
			   query_compress_fwd,query_compress_rev,/*genestrand*/0,
			   max_insertionlen,max_deletionlen,overall_end_distance,
			   nmismatches_allowed,/*paired_end_p*/false,/*lowp*/true,
			   intlistpool,uintlistpool,univcoordlistpool,listpool,pathpool,transcriptpool,
			   vectorpool,hitlistpool,spliceendsgen,
			   /*extend_qstart_p*/true,/*extend_qend_p*/true);
    if (completep == true) {
      debug(printf("Found extended_antisense_paths_gminus\n"));
      *antisense_paths_gminus = List_append(newpaths,*antisense_paths_gminus);
    } else {
      this->unextended_antisense_paths_gminus = List_append(newpaths,this->unextended_antisense_paths_gminus);
    }
  }

  return;
}


static List_T
single_read_fusion (int *found_score, T this, int querylength,

		    Compress_T query_compress_fwd, Compress_T query_compress_rev,
		    Shortread_T queryseq, Knownsplicing_T knownsplicing, 
		    int nmismatches_allowed, int max_insertionlen, int max_deletionlen,

		    Univdiagpool_T univdiagpool, Intlistpool_T intlistpool,
		    Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
		    Listpool_T listpool, Pathpool_T pathpool, Transcriptpool_T transcriptpool,
		    Vectorpool_T vectorpool, Hitlistpool_T hitlistpool) {

  List_T paths = NULL, p;
  Path_T path;

  /* Look for possible fusions (or combinations of existing paths) */
  /* Find fusions.  Use code similar to finding outer fusions in Pathpair_eval_and_sort */

  for (p = this->unextended_sense_paths_gplus; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);

    if (Path_unextended_qend_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qstart_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_querystart_plus(&(*found_score),/*main*/path,this,
						      query_compress_fwd,query_compress_rev,
						      queryseq,querylength,knownsplicing,/*genestrand*/0,
						      nmismatches_allowed,max_insertionlen,max_deletionlen,
						      intlistpool,uintlistpool,univcoordlistpool,
						      listpool,univdiagpool,pathpool,vectorpool,
						      transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
      
    if (Path_unextended_qstart_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qend_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_queryend_plus(&(*found_score),/*main*/path,this,
						    query_compress_fwd,query_compress_rev,
						    queryseq,querylength,knownsplicing,/*genestrand*/0,
						    nmismatches_allowed,max_insertionlen,max_deletionlen,
						    intlistpool,uintlistpool,univcoordlistpool,
						    listpool,univdiagpool,pathpool,vectorpool,
						    transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
  }
    
  for (p = this->unextended_antisense_paths_gplus; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);

    if (Path_unextended_qend_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qstart_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_querystart_plus(&(*found_score),/*main*/path,this,
						      query_compress_fwd,query_compress_rev,
						      queryseq,querylength,knownsplicing,/*genestrand*/0,
						      nmismatches_allowed,max_insertionlen,max_deletionlen,
						      intlistpool,uintlistpool,univcoordlistpool,
						      listpool,univdiagpool,pathpool,vectorpool,
						      transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
      
    if (Path_unextended_qstart_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qend_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_queryend_plus(&(*found_score),/*main*/path,this,
						    query_compress_fwd,query_compress_rev,
						    queryseq,querylength,knownsplicing,/*genestrand*/0,
						    nmismatches_allowed,max_insertionlen,max_deletionlen,
						    intlistpool,uintlistpool,univcoordlistpool,
						    listpool,univdiagpool,pathpool,vectorpool,
						    transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
  }
    
  for (p = this->unextended_sense_paths_gminus; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);

    if (Path_unextended_qend_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qstart_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_querystart_minus(&(*found_score),/*main*/path,this,
						       query_compress_fwd,query_compress_rev,
						       queryseq,querylength,knownsplicing,/*genestrand*/0,
						       nmismatches_allowed,max_insertionlen,max_deletionlen,
						       intlistpool,uintlistpool,univcoordlistpool,
						       listpool,univdiagpool,pathpool,vectorpool,
						       transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }

    if (Path_unextended_qstart_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qend_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_queryend_minus(&(*found_score),/*main*/path,this,
						     query_compress_fwd,query_compress_rev,
						     queryseq,querylength,knownsplicing,/*genestrand*/0,
						     nmismatches_allowed,max_insertionlen,max_deletionlen,
						     intlistpool,uintlistpool,univcoordlistpool,
						     listpool,univdiagpool,pathpool,vectorpool,
						     transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
  }
    
  for (p = this->unextended_antisense_paths_gminus; p != NULL; p = List_next(p)) {
    path = (Path_T) List_head(p);

    if (Path_unextended_qend_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qstart_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_querystart_minus(&(*found_score),/*main*/path,this,
						       query_compress_fwd,query_compress_rev,
						       queryseq,querylength,knownsplicing,/*genestrand*/0,
						       nmismatches_allowed,max_insertionlen,max_deletionlen,
						       intlistpool,uintlistpool,univcoordlistpool,
						       listpool,univdiagpool,pathpool,vectorpool,
						       transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
      
    if (Path_unextended_qstart_p(path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == false &&
	Path_unextended_qend_p(path,/*endtrim_allowed*/25,/*allow_ambig_p*/false) == true) {
      paths = List_append(Path_fusion_queryend_minus(&(*found_score),/*main*/path,this,
						     query_compress_fwd,query_compress_rev,
						     queryseq,querylength,knownsplicing,/*genestrand*/0,
						     nmismatches_allowed,max_insertionlen,max_deletionlen,
						     intlistpool,uintlistpool,univcoordlistpool,
						     listpool,univdiagpool,pathpool,vectorpool,
						     transcriptpool,hitlistpool,/*endtrim_allowed*/8),paths);
    }
  }

  return paths;
}


static void
convert_trpaths (int *found_score,

		 List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		 List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		 List_T sense_trpaths, List_T antisense_trpaths,
		 T this, Shortread_T queryseq, int querylength,
		 Compress_T query_compress_fwd, Compress_T query_compress_rev,
		 Uintlistpool_T uintlistpool, Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
		 Listpool_T listpool, Pathpool_T pathpool, Transcriptpool_T transcriptpool,
		 Hitlistpool_T hitlistpool) {
  bool solvedp = false;

  Trpath_convert_sense(&solvedp,&(*found_score),
		       &this->unsolved_sense_paths_gplus,&this->unsolved_sense_paths_gminus,
		       &(*sense_paths_gplus),&(*sense_paths_gminus),
		       sense_trpaths,query_compress_fwd,query_compress_rev,querylength,
		       intlistpool,univcoordlistpool,listpool,pathpool,
		       transcriptpool,hitlistpool);
  Trpath_convert_antisense(&solvedp,&(*found_score),
			   &this->unsolved_antisense_paths_gplus,&this->unsolved_antisense_paths_gminus,
			   &(*antisense_paths_gplus),&(*antisense_paths_gminus),
			   antisense_trpaths,query_compress_fwd,query_compress_rev,querylength,
			   intlistpool,univcoordlistpool,listpool,pathpool,
			   transcriptpool,hitlistpool);

  *sense_paths_gplus = Path_consolidate(*sense_paths_gplus,queryseq,
					query_compress_fwd,query_compress_rev,
					uintlistpool,intlistpool,univcoordlistpool,
					listpool,pathpool,transcriptpool,hitlistpool);
  *sense_paths_gminus = Path_consolidate(*sense_paths_gminus,queryseq,
					query_compress_fwd,query_compress_rev,
					uintlistpool,intlistpool,univcoordlistpool,
					listpool,pathpool,transcriptpool,hitlistpool);
  *antisense_paths_gplus = Path_consolidate(*antisense_paths_gplus,queryseq,
					   query_compress_fwd,query_compress_rev,
					   uintlistpool,intlistpool,univcoordlistpool,
					   listpool,pathpool,transcriptpool,hitlistpool);
  *antisense_paths_gminus = Path_consolidate(*antisense_paths_gminus,queryseq,
					    query_compress_fwd,query_compress_rev,
					    uintlistpool,intlistpool,univcoordlistpool,
					    listpool,pathpool,transcriptpool,hitlistpool);

  return;
}
    


bool
single_read (int *found_score, Method_T *last_method,

	     List_T *sense_paths_gplus, List_T *sense_paths_gminus,
	     List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

	     T this, EF64_T repetitive_ef64, int genestrand,

	     Shortread_T queryseq, char *queryuc_ptr, char *queryrc, int querylength,
	     Knownsplicing_T knownsplicing, Knownindels_T knownindels,
	     int *mismatch_positions_alloc, Univcoord_T *novel_diagonals_alloc, unsigned short *localdb_alloc,
	     Compress_T query_compress_fwd, Compress_T query_compress_rev,

	     int localdb_nmismatches_allowed, int max_insertionlen, int max_deletionlen,
	     Chrpos_T overall_max_distance, Chrpos_T overall_end_distance,
			 
	     Trdiagpool_T trdiagpool, Univdiagpool_T univdiagpool,
	     Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
	     Univcoordlistpool_T univcoordlistpool, Listpool_T listpool, 
	     Trpathpool_T trpathpool, Pathpool_T pathpool, Transcriptpool_T transcriptpool,
	     Vectorpool_T vectorpool, Hitlistpool_T hitlistpool, Spliceendsgen_T spliceendsgen,
	     bool paired_end_p, bool first_read_p, Method_T final_univdiagonal_method) {

  bool solvedp = false;
  int sufficient_score = querylength/20;
  int i;

  Univcoord_T *univdiagonals_gplus, *univdiagonals_gminus;
  int nunivdiagonals_gplus, nunivdiagonals_gminus;
    
  Trpath_T trpath;
  int tstart, tend;
  int index;


  *sense_paths_gplus = NULL;
  *sense_paths_gminus = NULL;
  *antisense_paths_gplus = NULL;
  *antisense_paths_gminus = NULL;
    
  if (transcriptome_align_p == true) {
    /* A.  Transcriptome search using univdiagonals */
    while (*found_score > sufficient_score && *last_method < TR_ANYPAIR) {
      /* Append results to lists in Stage1_T object */
      *last_method = single_read_next_method_trdiagonal(*last_method,this,querylength);
      tstart = 0;
      tend = querylength;
      for (index = 0; index < this->n_sense_trdiagonals; index++) {
	if (this->sense_tstarts != NULL) {
	  tstart = this->sense_tstarts[index];
	  tend = this->sense_tends[index];
	}
	if ((trpath = Trpath_solve_from_trdiagonal(&(*found_score),/*trdiagonal*/this->sense_trdiagonals[index],
						   tstart,tend,/*trnum*/this->sense_trnums[index],
						   /*troffset*/this->sense_troffsets[index],
						   /*trhigh*/this->sense_trhighs[index],
						   /*query_compress_tr*/query_compress_fwd,/*tplusp*/true,querylength,
						   mismatch_positions_alloc,/*want_lowest_coordinate_p*/true,
						   this->indelinfo,max_insertionlen,max_deletionlen,
						   intlistpool,uintlistpool,listpool,trpathpool,pathpool,
						   *last_method)) != NULL) {
	  this->sense_trpaths = Hitlist_push(this->sense_trpaths,hitlistpool,(void *) trpath
					     hitlistpool_trace(__FILE__,__LINE__));
	}
      }

      tstart = 0;
      tend = querylength;
      for (index = 0; index < this->n_antisense_trdiagonals; index++) {
	if (this->antisense_tstarts != NULL) {
	  tstart = this->antisense_tstarts[index];
	  tend = this->antisense_tends[index];
	}
	if ((trpath = Trpath_solve_from_trdiagonal(&(*found_score),/*trdiagonal*/this->antisense_trdiagonals[index],
						   tstart,tend,/*trnum*/this->antisense_trnums[index],
						   /*troffset*/this->antisense_troffsets[index],
						   /*trhigh*/this->antisense_trhighs[index],
						   /*query_compress_tr*/query_compress_rev,/*tplusp*/false,querylength,
						   mismatch_positions_alloc,/*want_lowest_coordinate_p*/true,
						   this->indelinfo,max_insertionlen,max_deletionlen,
						   intlistpool,uintlistpool,listpool,trpathpool,pathpool,
						   *last_method)) != NULL) {
	  this->antisense_trpaths = Hitlist_push(this->antisense_trpaths,hitlistpool,(void *) trpath
						 hitlistpool_trace(__FILE__,__LINE__));
	}
      }
    }

    if (*found_score <= sufficient_score) {
      convert_trpaths(&(*found_score),
		      
		      &(*sense_paths_gplus),&(*sense_paths_gminus),
		      &(*antisense_paths_gplus),&(*antisense_paths_gminus),

		      this->sense_trpaths,this->antisense_trpaths,
		      this,queryseq,querylength,query_compress_fwd,query_compress_rev,
		      uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
		      transcriptpool,hitlistpool);
      return true;
    }


    /* B.  Transcriptome search using trpaths */
    while (*found_score > sufficient_score && *last_method < TR_EXT) {
      /* Append results to lists in Stage1_T object */
      *last_method = single_read_next_method_tr(&(*found_score),*last_method,
					     
					       &this->sense_trpaths,&this->antisense_trpaths,
					     
					       this,genestrand,querylength,
					       mismatch_positions_alloc,
					       query_compress_fwd,query_compress_rev,
					       localdb_nmismatches_allowed,max_insertionlen,max_deletionlen,
					       trdiagpool,intlistpool,uintlistpool,
					       listpool,trpathpool,pathpool,hitlistpool,
					       /*appendp*/true);
    }

    if (*found_score <= sufficient_score) {
      convert_trpaths(&(*found_score),

		      &(*sense_paths_gplus),&(*sense_paths_gminus),
		      &(*antisense_paths_gplus),&(*antisense_paths_gminus),

		      this->sense_trpaths,this->antisense_trpaths,
		      this,queryseq,querylength,query_compress_fwd,query_compress_rev,
		      uintlistpool,intlistpool,univcoordlistpool,listpool,pathpool,
		      transcriptpool,hitlistpool);
      return true;
    }
    

    /* C.  Prep for genome search.  Convert all trpaths to paths */
    Trpath_convert_sense(&solvedp,&(*found_score),
			 &this->unsolved_sense_paths_gplus,&this->unsolved_sense_paths_gminus,
			 &(*sense_paths_gplus),&(*sense_paths_gminus),
			 this->sense_trpaths,query_compress_fwd,query_compress_rev,querylength,
			 intlistpool,univcoordlistpool,listpool,pathpool,
			 transcriptpool,hitlistpool);
    Trpath_convert_antisense(&solvedp,&(*found_score),
			     &this->unsolved_antisense_paths_gplus,&this->unsolved_antisense_paths_gminus,
			     &(*antisense_paths_gplus),&(*antisense_paths_gminus),
			     this->antisense_trpaths,query_compress_fwd,query_compress_rev,querylength,
			     intlistpool,univcoordlistpool,listpool,pathpool,
			     transcriptpool,hitlistpool);
    if (*found_score <= sufficient_score) {
      *sense_paths_gplus = Path_consolidate(*sense_paths_gplus,queryseq,
					    query_compress_fwd,query_compress_rev,
					    uintlistpool,intlistpool,univcoordlistpool,
					    listpool,pathpool,transcriptpool,hitlistpool);
      *sense_paths_gminus = Path_consolidate(*sense_paths_gminus,queryseq,
					     query_compress_fwd,query_compress_rev,
					     uintlistpool,intlistpool,univcoordlistpool,
					     listpool,pathpool,transcriptpool,hitlistpool);
      *antisense_paths_gplus = Path_consolidate(*antisense_paths_gplus,queryseq,
						query_compress_fwd,query_compress_rev,
						uintlistpool,intlistpool,univcoordlistpool,
						listpool,pathpool,transcriptpool,hitlistpool);
      *antisense_paths_gminus = Path_consolidate(*antisense_paths_gminus,queryseq,
						 query_compress_fwd,query_compress_rev,
						 uintlistpool,intlistpool,univcoordlistpool,
						 listpool,pathpool,transcriptpool,hitlistpool);
      return true;
    }

    /* D.  Try unsolved trpaths, if any */
    debug(printf("Read:  Solving unsolved trpaths\n"));
    single_read_unsolved_tr(&(*found_score),
			  
			    &(*sense_paths_gplus),&(*sense_paths_gminus),
			    &(*antisense_paths_gplus),&(*antisense_paths_gminus),

			    this,queryseq,querylength,knownsplicing,
			    query_compress_fwd,query_compress_rev,
			    max_insertionlen,max_deletionlen,
			    intlistpool,uintlistpool,univcoordlistpool,
			    listpool,pathpool,transcriptpool,hitlistpool);

    if (*found_score <= sufficient_score) {
      return true;
    }
  }

  /* RNA or DNA task */    
  /* Solving using univdiagonals */

  /* Kmer_prevalent is slow */
  while (*found_score > sufficient_score && *last_method < final_univdiagonal_method) {
    *last_method = single_read_next_method_univdiagonal(*last_method,
							&univdiagonals_gplus,&nunivdiagonals_gplus,
							&univdiagonals_gminus,&nunivdiagonals_gminus,
							this,/*mate*/NULL,genestrand,querylength,
							overall_end_distance,first_read_p);
    
    for (i = 0; i < nunivdiagonals_gplus; i++) {
      Path_solve_from_univdiagonal(&(*found_score),

				   &this->unextended_sense_paths_gplus,&this->unextended_antisense_paths_gplus,
				   &(*sense_paths_gplus),&(*antisense_paths_gplus),
				   
				   /*univdiagonal*/univdiagonals_gplus[i],
				   queryseq,/*queryptr*/queryuc_ptr,/*query_compress*/query_compress_fwd,
				   query_compress_fwd,query_compress_rev,
				   /*plusp*/true,querylength,mismatch_positions_alloc,
				   
				   novel_diagonals_alloc,localdb_alloc,this,knownsplicing,knownindels,
				   max_insertionlen,max_deletionlen,localdb_nmismatches_allowed,
				   overall_end_distance,paired_end_p,first_read_p,

				   intlistpool,uintlistpool,univcoordlistpool,
				   listpool,pathpool,transcriptpool,vectorpool,hitlistpool,spliceendsgen,
				   /*method*/*last_method,/*find_splices_p*/true);
    }

    for (i = 0; i < nunivdiagonals_gminus; i++) {
      Path_solve_from_univdiagonal(&(*found_score),

				   &this->unextended_sense_paths_gminus,&this->unextended_antisense_paths_gminus,
				   &(*sense_paths_gminus),&(*antisense_paths_gminus),
				   
				   /*univdiagonal*/univdiagonals_gminus[i],
				   queryseq,/*queryptr*/queryrc,/*query_compress*/query_compress_rev,
				   query_compress_fwd,query_compress_rev,
				   /*plusp*/false,querylength,mismatch_positions_alloc,

				   novel_diagonals_alloc,localdb_alloc,this,knownsplicing,knownindels,
				   max_insertionlen,max_deletionlen,localdb_nmismatches_allowed,
				   overall_end_distance,paired_end_p,first_read_p,

				   intlistpool,uintlistpool,univcoordlistpool,
				   listpool,pathpool,transcriptpool,vectorpool,hitlistpool,spliceendsgen,
				   /*method*/*last_method,/*find_splices_p*/true);
    }
    
    FREE_ALIGN(univdiagonals_gplus);
    FREE_ALIGN(univdiagonals_gminus);

    if (*found_score <= sufficient_score) {
      return true;
    }
  }

  if (*found_score <= sufficient_score) {
    return true;
  } else if (paired_end_p == true) {
    /* For paired-end reads, try fastest methods on each side initially */
    return false;
  }

  /* Solving using paths */
  while (*found_score > sufficient_score && *last_method < EXT) {
    *last_method = single_read_next_method_gen(&(*found_score),*last_method,
						
					       &(*sense_paths_gplus),&(*sense_paths_gminus),
					       &(*antisense_paths_gplus),&(*antisense_paths_gminus),
						
					       this,repetitive_ef64,genestrand,
					       queryseq,queryuc_ptr,queryrc,querylength,
					       knownsplicing,knownindels,mismatch_positions_alloc,
					       novel_diagonals_alloc,localdb_alloc,
					       query_compress_fwd,query_compress_rev,
					       localdb_nmismatches_allowed,max_insertionlen,max_deletionlen,
					       overall_max_distance,overall_end_distance,
					       univdiagpool,intlistpool,uintlistpool,univcoordlistpool,
					       listpool,pathpool,transcriptpool,vectorpool,
					       hitlistpool,spliceendsgen,
					       paired_end_p,first_read_p,/*appendp*/true);
  }

  if (*found_score <= sufficient_score) {
    return true;
  }


  /* Perform extension */
  /* Extend procedures used with single-end reads, since no paired-end
     to help with extension */
  debug(printf("Read: 5.  Performing extension\n"));
  single_read_extend(&(*found_score),this,

		     &(*sense_paths_gplus),&(*sense_paths_gminus),
		     &(*antisense_paths_gplus),&(*antisense_paths_gminus),

		     queryseq,queryuc_ptr,queryrc,querylength,
		     knownsplicing,knownindels,mismatch_positions_alloc,
		     novel_diagonals_alloc,localdb_alloc,
		     query_compress_fwd,query_compress_rev,
		     localdb_nmismatches_allowed,max_insertionlen,max_deletionlen,
		     overall_end_distance,
		     intlistpool,uintlistpool,univcoordlistpool,
		     listpool,pathpool,transcriptpool,vectorpool,
		     hitlistpool,spliceendsgen);

  if (*found_score <= sufficient_score) {
    return true;
  } else {
    return false;
  }
}



Path_T *
Stage1_single_read (int *npaths_primary, int *npaths_altloc, int *first_absmq, int *second_absmq,
		    Shortread_T queryseq, EF64_T repetitive_ef64,
		    Knownsplicing_T knownsplicing, Knownindels_T knownindels,
		    Trdiagpool_T trdiagpool, Univdiagpool_T univdiagpool,
		    Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		    Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
		    Trpathpool_T trpathpool, Pathpool_T pathpool, Transcriptpool_T transcriptpool,
		    Vectorpool_T vectorpool, Hitlistpool_T hitlistpool, 
		    Spliceendsgen_T spliceendsgen, bool single_cell_p, Pass_T pass) {
  Path_T *patharray, path;
  T this;
  List_T paths;

  List_T sense_paths_gplus, sense_paths_gminus,
    antisense_paths_gplus, antisense_paths_gminus;

  int *mismatch_positions_alloc;
  Univcoord_T *novel_diagonals_alloc;
  unsigned short *localdb_alloc;

  int nmismatches_filter, mincoverage_filter;
  int nmismatches_allowed;
  int max_middle_insertions, max_middle_deletions, max_insertionlen, max_deletionlen;
  Chrpos_T overall_max_distance, overall_end_distance;

  int querylength;
  char *queryuc_ptr, *queryrc;
  Compress_T query_compress_fwd, query_compress_rev;

  int found_score;
  Method_T last_method, final_univdiagonal_method;
  int i;
  
#if 0
  bool first_read_p;
  if (single_cell_p == true) {
    first_read_p = false;
  } else {
    first_read_p = true;
  }
#endif

  if ((querylength = Shortread_fulllength(queryseq)) < index1part + index1interval - 1) {
    *npaths_primary = *npaths_altloc = 0;
    return (Path_T *) NULL;
  } else {
    queryuc_ptr = Shortread_queryuc_ptr(queryseq);
    queryrc = Shortread_queryrc(queryseq);
    this = Stage1_new(queryuc_ptr,querylength);    
  }

  /* nmismatches_allowed means nmismatches_search and is not specified
     by the user.  The user-specified value for -m represents
     nmismatches_filter */
  /* TODO: make this dependent upon the defect_rate */
  nmismatches_allowed = querylength/20; /* was querylength/index1part */

  if (user_nmismatches_filter_float < 0.0) {
    /* Not specified, so don't filter */
    nmismatches_filter = querylength;
  } else if (user_nmismatches_filter_float < 1.0) {
    nmismatches_filter = (int) rint(user_nmismatches_filter_float * (double) querylength);
  } else {
    nmismatches_filter = (int) user_nmismatches_filter_float;
  }

  if (user_mincoverage_filter_float <= 0.0) {
    mincoverage_filter = 0;
  } else if (user_mincoverage_filter_float <= 1.0) {
    /* Assuming that --min-coverage=1 must mean 1.0 and not a coverage of 1 bp */
    mincoverage_filter = (int) rint(user_mincoverage_filter_float * (double) querylength);
  } else {
    mincoverage_filter = (int) user_mincoverage_filter_float;
  }

  if (max_middle_insertions_float > 0.0 && max_middle_insertions_float < 1.0) {
    max_middle_insertions = (int) rint(max_middle_insertions_float * (double) querylength);
  } else {
    max_middle_insertions = (int) max_middle_insertions_float;
  }
  max_insertionlen = max_middle_insertions;
  if (max_insertionlen > querylength) {
    max_insertionlen = querylength;
  }

  if (max_middle_deletions_float > 0.0 && max_middle_deletions_float < 1.0) {
    max_middle_deletions = (int) rint(max_middle_deletions_float * (double) querylength);
  } else {
    max_middle_deletions = (int) max_middle_deletions_float;
  }
  max_deletionlen = max_middle_deletions;

  overall_max_distance = shortsplicedist;
  if ((Chrpos_T) max_middle_deletions > overall_max_distance) {
    overall_max_distance = (Chrpos_T) max_middle_deletions;
  }
  if ((Chrpos_T) max_middle_insertions > overall_max_distance) {
    overall_max_distance = (Chrpos_T) max_middle_insertions;
  }
  overall_end_distance = shortsplicedist_novelend > (Chrpos_T) max_deletionlen ? shortsplicedist_novelend : (Chrpos_T) max_deletionlen;


  mismatch_positions_alloc = (int *) MALLOC((querylength+MISMATCH_EXTRA)*sizeof(int));

  /* 2 localdb regions possible if shortsplicedist_novelend < 65536 */
  /* 65536 represents the worst possible case where every position in the localdb region matches the query */
  novel_diagonals_alloc = (Univcoord_T *) MALLOC(2 * 65536 *sizeof(Univcoord_T));
  MALLOC_ALIGN(localdb_alloc,65536 * sizeof(unsigned short)); /* Maximum number of intersections in a localdb region */

  query_compress_fwd = Compress_new_fwd(queryuc_ptr,querylength);
  query_compress_rev = Compress_new_rev(queryuc_ptr,querylength);

  if (splicingp == false) {
    final_univdiagonal_method = KMER_EXACT1;
  } else {
    final_univdiagonal_method = KMER_EXACT2;
  }

  if (mode == STANDARD || mode == CMET_STRANDED || mode == ATOI_STRANDED || mode == TTOC_STRANDED) {
    found_score = querylength;
    last_method = METHOD_INIT;
    if (single_read(&found_score,&last_method,

		    &sense_paths_gplus,&sense_paths_gminus,
		    &antisense_paths_gplus,&antisense_paths_gminus,

		    this,repetitive_ef64,/*genestrand*/0,
		    queryseq,queryuc_ptr,queryrc,querylength,
		    knownsplicing,knownindels,mismatch_positions_alloc,
		    novel_diagonals_alloc,localdb_alloc,
		    query_compress_fwd,query_compress_rev,

		    nmismatches_allowed,max_insertionlen,max_deletionlen,
		    overall_max_distance,overall_end_distance,
		    trdiagpool,univdiagpool,intlistpool,uintlistpool,univcoordlistpool,
		    listpool,trpathpool,pathpool,transcriptpool,vectorpool,
		    hitlistpool,spliceendsgen,
		    /*paired_end_p*/false,/*first_read_p*/true,final_univdiagonal_method) == true) {
      paths = List_append(sense_paths_gplus,
			  List_append(sense_paths_gminus,
				      List_append(antisense_paths_gplus,
						  antisense_paths_gminus)));
    } else if (splicingp == false) {
      paths = List_append(sense_paths_gplus,
			  List_append(sense_paths_gminus,
				      List_append(antisense_paths_gplus,
						  antisense_paths_gminus)));

    } else {
      /* Perform fusions, if single-end and splicingp is true */
      debug(printf("Read: 6.  Performing fusions\n"));
      Stage1_extend_unextended_paths(this,queryseq,queryuc_ptr,queryrc,querylength,/*genestrand*/0,
				     query_compress_fwd,query_compress_rev,
				     knownsplicing,knownindels,
				     mismatch_positions_alloc,novel_diagonals_alloc,localdb_alloc,
				     nmismatches_allowed,max_insertionlen,max_deletionlen,
				     overall_end_distance,/*paired_end_p*/false,/*first_read_p*/true,
				     intlistpool,uintlistpool,univcoordlistpool,
				     listpool,pathpool,transcriptpool,
				     vectorpool,hitlistpool,spliceendsgen);

      paths = single_read_fusion(&found_score,this,querylength,
				 query_compress_fwd,query_compress_rev,queryseq,knownsplicing,
				 nmismatches_allowed,max_insertionlen,max_deletionlen,
				 univdiagpool,intlistpool,uintlistpool,univcoordlistpool,
				 listpool,pathpool,transcriptpool,vectorpool,hitlistpool);
    }
  }

  if (paths != NULL) {
    Path_gc(&this->unextended_sense_paths_gplus,intlistpool,univcoordlistpool,listpool,
	    pathpool,transcriptpool,hitlistpool);
    Path_gc(&this->unextended_sense_paths_gminus,intlistpool,univcoordlistpool,listpool,
	    pathpool,transcriptpool,hitlistpool);
    Path_gc(&this->unextended_antisense_paths_gplus,intlistpool,univcoordlistpool,listpool,
	    pathpool,transcriptpool,hitlistpool);
    Path_gc(&this->unextended_antisense_paths_gminus,intlistpool,univcoordlistpool,listpool,
	    pathpool,transcriptpool,hitlistpool);

  } else {
    /* As last resort, use unextended paths */
     /* Should have called single_read_extend, which means no further extensions are possible */
    paths = List_append(this->unextended_sense_paths_gplus,
			List_append(this->unextended_sense_paths_gminus,
				    List_append(this->unextended_antisense_paths_gplus,
						this->unextended_antisense_paths_gminus)));
    this->unextended_sense_paths_gplus = (List_T) NULL;
    this->unextended_sense_paths_gminus = (List_T) NULL;
    this->unextended_antisense_paths_gplus = (List_T) NULL;
    this->unextended_antisense_paths_gminus = (List_T) NULL;
  }
    
  if (paths == NULL) {
    *npaths_primary = *npaths_altloc = 0;
    patharray = (Path_T *) NULL;

  } else {
    patharray = (Path_T *) List_to_array_out(paths,NULL);
    patharray = Path_eval_and_sort(&(*npaths_primary),&(*npaths_altloc),
				   &(*first_absmq),&(*second_absmq),patharray,
				   /*npaths*/List_length(paths),
				   query_compress_fwd,query_compress_rev,queryuc_ptr,queryrc,
				   Shortread_quality_string(queryseq),nmismatches_filter,mincoverage_filter,
				   intlistpool,univcoordlistpool,listpool,
				   pathpool,transcriptpool,hitlistpool);

    if (transcriptome != NULL && pass == PASS2) {
      for (i = 0; i < (*npaths_primary) + (*npaths_altloc); i++) {
	path = patharray[i];
	Transcript_velocity_single(path);
      }
    }

    Hitlistpool_free_list(&paths,hitlistpool
			  hitlistpool_trace(__FILE__,__LINE__));
  }
  
  Compress_free(&query_compress_fwd);
  Compress_free(&query_compress_rev);
  FREE_ALIGN(localdb_alloc);
  FREE(novel_diagonals_alloc);
  FREE(mismatch_positions_alloc);

  /* Do not free paths, since they are now appended to paths */
  Stage1_free(&this,trdiagpool,univdiagpool,intlistpool,uintlistpool,
	      univcoordlistpool,listpool,pathpool,trpathpool,
	      transcriptpool,hitlistpool,/*free_paths_p*/false);

  /* FREE(queryrc); -- Now taken from Shortread */

  return patharray;
}


void
Stage1hr_single_setup (Mode_T mode_in, int index1part_in, int index1interval_in, int index1part_tr_in,
		       Transcriptome_T transcriptome_in, bool genome_align_p_in,
		       bool transcriptome_align_p_in, 
		       double user_nmismatches_filter_float_in, double user_mincoverage_filter_float_in,
		       double max_middle_insertions_float_in, double max_middle_deletions_float_in,
		       bool splicingp_in, Chrpos_T shortsplicedist_in, Chrpos_T shortsplicedist_novelend_in) {

  mode = mode_in;
  index1part = index1part_in;
  index1interval = index1interval_in;
  index1part_tr = index1part_tr_in;

  transcriptome = transcriptome_in;
  genome_align_p = genome_align_p_in;
  transcriptome_align_p = transcriptome_align_p_in;

  user_nmismatches_filter_float = user_nmismatches_filter_float_in;
  user_mincoverage_filter_float = user_mincoverage_filter_float_in;

  max_middle_insertions_float = max_middle_insertions_float_in;
  max_middle_deletions_float = max_middle_deletions_float_in;

  splicingp = splicingp_in;
  shortsplicedist = shortsplicedist_in;
  shortsplicedist_novelend = shortsplicedist_novelend_in;

  return;
}
