static char rcsid[] = "$Id: d52e557248e5390e527f3dfbe6c6c509567a613c $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif

#include "trpath.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "mem.h"
#include "assert.h"


#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* Allocation and freeing of objects */
#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif


/* Printing */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif


/* endpoints_acceptable_p */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif


/* Conversion to Path_T */
#ifdef DEBUG10
#define debug10(x) x
#else
#define debug10(x)
#endif


#define T Trpath_T

T
Trpath_create (Intlist_T endpoints, Uintlist_T trdiagonals, Intlist_T nmismatches,
	       List_T junctions, bool tplusp,
	       Trnum_T trnum, Trcoord_T troffset, Trcoord_T trhigh,
	       Trpathpool_T trpathpool) {

  T new = Trpathpool_new_trpath(trpathpool
				trpathpool_trace(__FILE__,__LINE__));
  Intlist_T q;
#ifdef CHECK_ASSERTIONS
  int prev_endpoint;
#endif

  debug0(printf("Creating path %p by Trpath_create\n",new));
  
  new->tplusp = tplusp;

  new->trnum = trnum;
  new->troffset = troffset;
  new->trhigh = trhigh;

  new->endpoints = endpoints;
  new->trdiagonals = trdiagonals;
  new->nmismatches = nmismatches;
  new->junctions = junctions;

#ifdef CHECK_ASSERTIONS
  prev_endpoint = Intlist_head(new->endpoints);
  for (q = Intlist_next(new->endpoints); q != NULL; q = Intlist_next(q)) {
    if (Intlist_head(q) <= prev_endpoint) {
      printf("Trpath_create expected forward, but got\n");
      Trpath_print(new);
      abort();
    }
    prev_endpoint = Intlist_head(q);
  }
#endif

  return new;
}


void
Trpath_free (T *old, Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
	     Listpool_T listpool, Trpathpool_T trpathpool, Pathpool_T pathpool) {
  Intlistpool_free_list(&(*old)->endpoints,intlistpool
			intlistpool_trace(__FILE__,__LINE__));
  Uintlistpool_free_list(&(*old)->trdiagonals,uintlistpool
			  uintlistpool_trace(__FILE__,__LINE__));
  Intlistpool_free_list(&(*old)->nmismatches,intlistpool
			intlistpool_trace(__FILE__,__LINE__));
  Junction_list_gc(&(*old)->junctions,listpool,pathpool);
  Trpathpool_free_trpath(&(*old),trpathpool
			 trpathpool_trace(__FILE__,__LINE__));
  return;
}


T
Trpath_new_for_tstart_extension (Trcoord_T trdiagonal, int tstart, int tend, int nmismatches,
				 bool tplusp, Trnum_T trnum, Trcoord_T troffset, Trcoord_T trhigh,
				 Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
				 Trpathpool_T trpathpool) {
  T new = Trpathpool_new_trpath(trpathpool
				trpathpool_trace(__FILE__,__LINE__));

  new->tplusp = tplusp;

  new->trnum = trnum;
  new->troffset = troffset;
  new->trhigh = trhigh;

  new->endpoints = Intlistpool_push(Intlistpool_push(NULL,intlistpool,tend
						     intlistpool_trace(__FILE__,__LINE__)),intlistpool,tstart
				    intlistpool_trace(__FILE__,__LINE__));
  new->trdiagonals = Uintlistpool_push(NULL,uintlistpool,trdiagonal
				        uintlistpool_trace(__FILE__,__LINE__));
  new->nmismatches = Intlistpool_push(NULL,intlistpool,nmismatches
				      intlistpool_trace(__FILE__,__LINE__)); /* nmismatches_to_trimpos */
  new->junctions = (List_T) NULL;
  
  return new;
}


T
Trpath_new_for_tend_extension (Trcoord_T trdiagonal, int tstart, int tend, int nmismatches,
			       bool tplusp, Trnum_T trnum, Trcoord_T troffset, Trcoord_T trhigh,
			       Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			       Trpathpool_T trpathpool) {
  T new = Trpathpool_new_trpath(trpathpool
				trpathpool_trace(__FILE__,__LINE__));

  new->tplusp = tplusp;

  new->trnum = trnum;
  new->troffset = troffset;
  new->trhigh = trhigh;

  new->endpoints = Intlistpool_push(Intlistpool_push(NULL,intlistpool,tstart
						     intlistpool_trace(__FILE__,__LINE__)),intlistpool,tend
				    intlistpool_trace(__FILE__,__LINE__));
  new->trdiagonals = Uintlistpool_push(NULL,uintlistpool,trdiagonal
				        uintlistpool_trace(__FILE__,__LINE__));
  new->nmismatches = Intlistpool_push(NULL,intlistpool,nmismatches
				      intlistpool_trace(__FILE__,__LINE__));
  new->junctions = (List_T) NULL;

  return new;
}


T
Trpath_new_from_ends (Trcoord_T trdiagonal5, int tstart5, int tend5,
		      Trcoord_T trdiagonal3, int tstart3, int tend3,
		      bool tplusp, Trnum_T trnum, Trcoord_T troffset, Trcoord_T trhigh,
		      Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		      Listpool_T listpool, Trpathpool_T trpathpool) {
  (void)(tend5);

  T new = Trpathpool_new_trpath(trpathpool
				trpathpool_trace(__FILE__,__LINE__));
  debug0(printf("Creating trpath %p by Trpath_new_from_ends\n",new));
  
  new->tplusp = tplusp;
  new->trnum = trnum;
  new->troffset = troffset;
  new->trhigh = trhigh;

  new->endpoints = Intlistpool_push(NULL,intlistpool,tend3
				    intlistpool_trace(__FILE__,__LINE__));
  new->endpoints = Intlistpool_push(new->endpoints,intlistpool,tstart3
				    intlistpool_trace(__FILE__,__LINE__));
  new->endpoints = Intlistpool_push(new->endpoints,intlistpool,tstart5
				    intlistpool_trace(__FILE__,__LINE__));

  new->trdiagonals = Uintlistpool_push(NULL,uintlistpool,trdiagonal3
				        uintlistpool_trace(__FILE__,__LINE__));
  new->trdiagonals = Uintlistpool_push(new->trdiagonals,uintlistpool,trdiagonal5
				        uintlistpool_trace(__FILE__,__LINE__));

  new->nmismatches = Intlistpool_push(NULL,intlistpool,-1
				      intlistpool_trace(__FILE__,__LINE__));
  new->nmismatches = Intlistpool_push(new->nmismatches,intlistpool,-1
				      intlistpool_trace(__FILE__,__LINE__));

#ifdef ALLOCATE_UNSOLVED_JUNCTION
  new->junctions = Listpool_push(NULL,listpool,Junction_new_unsolved(pathpool)
				 listpool_trace(__FILE__,__LINE__));
#else
  new->junctions = Listpool_push(NULL,listpool,(void *) JUNCTION_UNSOLVED
				 listpool_trace(__FILE__,__LINE__));
#endif
  
  return new;
}


T
Trpath_new_exact (Trcoord_T trdiagonal, int tstart, int tend, int nmismatches,
		  bool tplusp, Trnum_T trnum, Trcoord_T troffset, Trcoord_T trhigh,
		  Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		  Trpathpool_T trpathpool) {

  T new = Trpathpool_new_trpath(trpathpool
				trpathpool_trace(__FILE__,__LINE__));
  debug0(printf("Creating trpath %p by Trpath_new_exact\n",new));
  
  new->tplusp = tplusp;
  new->trnum = trnum;
  new->troffset = troffset;
  new->trhigh = trhigh;

  new->endpoints = Intlistpool_push(NULL,intlistpool,tend
				    intlistpool_trace(__FILE__,__LINE__));
  new->endpoints = Intlistpool_push(new->endpoints,intlistpool,tstart
				    intlistpool_trace(__FILE__,__LINE__));

  new->trdiagonals = Uintlistpool_push(NULL,uintlistpool,trdiagonal
				        uintlistpool_trace(__FILE__,__LINE__));
  new->nmismatches = Intlistpool_push(NULL,intlistpool,nmismatches
				      intlistpool_trace(__FILE__,__LINE__));
  new->junctions = (List_T) NULL;
  
  return new;
}


T
Trpath_reverse (T this, bool expect_fwd_p) {
  Intlist_T q;
#ifdef CHECK_ASSERTIONS
  int prev_endpoint;
#endif

  this->endpoints = Intlist_reverse(this->endpoints);
  this->trdiagonals = Uintlist_reverse(this->trdiagonals);
  this->nmismatches = Intlist_reverse(this->nmismatches);
  this->junctions = List_reverse(this->junctions);

#ifdef CHECK_ASSERTIONS
  if (expect_fwd_p == true) {
    prev_endpoint = Intlist_head(this->endpoints);
    for (q = Intlist_next(this->endpoints); q != NULL; q = Intlist_next(q)) {
      if (Intlist_head(q) <= prev_endpoint) {
	printf("Expecting forward, but got\n");
	Trpath_print(this);
	abort();
      }
      prev_endpoint = Intlist_head(q);
    }
 
 } else {
    prev_endpoint = Intlist_head(this->endpoints);
    for (q = Intlist_next(this->endpoints); q != NULL; q = Intlist_next(q)) {
      if (Intlist_head(q) >= prev_endpoint) {
	printf("Expecting reverse, but got\n");
	Trpath_print(this);
	abort();
      }
      prev_endpoint = Intlist_head(q);
    }
  }
#endif

  return this;
}


#if defined(CHECK_ASSERTIONS) || defined(DEBUG1)
void
Trpath_print (T this) {

  if (this != NULL) {
    printf(">> Trpath %p: ",this);

    /* Info added for new version */
    printf("tplusp:%d ",this->tplusp);

    printf("%s  %d:%s  %s  nmismatches:%s  ",
	   Uintlist_to_string(this->trdiagonals),this->trnum,
	   Uintlist_to_string_offset(this->trdiagonals,this->troffset),
	   Intlist_to_string(this->endpoints),Intlist_to_string(this->nmismatches));

    printf("jcns:");
    Junction_print_list(this->junctions);

    printf("\n");
  }

  return;
}
#endif


/* The same as Path_endpoints_acceptable_p */
bool
Trpath_endpoints_acceptable_p (Intlist_T endpoints, List_T junctions) {
  Intlist_T p;
  List_T q;
  Junction_T junction;
  int last_endpoint;

  debug2(printf("Evaluating endpoints for acceptability: %s\n",Intlist_to_string(endpoints)));

  /* last_endpoint = 0; */
  /* Skip first endpoint */
  for (p = Intlist_next(endpoints), q = junctions; Intlist_next(p) != NULL; p = Intlist_next(p), q = List_next(q)) {
    last_endpoint = Intlist_head(p);
    junction = (Junction_T) List_head(q);
    if (last_endpoint + Junction_ninserts(junction) >= Intlist_head(Intlist_next(p))) {
      debug2(printf("Endpoint %d + %d >= %d, so unacceptable\n",
		    last_endpoint,Junction_ninserts(junction),Intlist_head(Intlist_next(p))));
      return false;
    } else {
      debug2(printf("Endpoint %d + %d < %d, so acceptable\n",
		    last_endpoint,Junction_ninserts(junction),Intlist_head(Intlist_next(p))));
    }
  }

  return true;
}


Path_T
Trpath_convert_to_path_geneplus (T trpath, int *exonbounds, Chrpos_T *exonstarts, int nexons,
				 Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh,
				 bool tplusp, int sensedir, int querylength,
				 Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
				 Listpool_T listpool, Pathpool_T pathpool, Method_T method) {
  bool gplusp;
  Intlist_T endpoints;
  Univcoordlist_T univdiagonals = NULL;
  Intlist_T nmismatches = NULL, ref_nmismatches = NULL;
  List_T junctions = NULL;

  Univcoord_T univdiagonal;
  Chrpos_T splice_distance;
  Junction_T between_junction = NULL;

  int exoni;
  int exstart, exend;
  int trstart, trend, tstart, tend;
  int exonpos, exonlength, uselength;

  Intlist_T tr_endpoints = trpath->endpoints;
  Uintlist_T trdiagonals = trpath->trdiagonals;
  Intlist_T tr_nmismatches = trpath->nmismatches;
  List_T tr_junctions = trpath->junctions;
  Trcoord_T troffset = trpath->troffset;
  Trcoord_T trdiagonal;


#ifdef DEBUG10
  printf("\nEntered Trpath_convert_to_path_geneplus\n");
  printf("tr_endpoints: %s\n",Intlist_to_string(tr_endpoints));
  printf("trdiagonals: %s\n",Uintlist_to_string(trdiagonals));
  printf("tr_nmismatches: %s\n",Intlist_to_string(tr_nmismatches));
  for (exoni = 0; exoni < nexons; exoni++) {
    printf("exon %d: exonbound %d, exonstart %u\n",exoni,exonbounds[exoni],exonstarts[exoni]);
  }
#endif

  /* tplusp (direction of read along transcript) and gplusp (direction of transcript along genome)
     combine to give plusp (direction of read along genome) */
  gplusp = (tplusp == true) ? true : false;

  /* Initial endpoint */
  tstart = Intlist_head(tr_endpoints);
  tend = Intlist_second_value(tr_endpoints);
  endpoints = Intlistpool_push(NULL,intlistpool,tstart
			       intlistpool_trace(__FILE__,__LINE__)); /* gplus */

  exoni = 0;
  exstart = 0;
  while (exoni < nexons && trdiagonals != NULL) {
    exend = exonbounds[exoni];
    trdiagonal = Uintlist_head(trdiagonals);
    trstart = (trdiagonal - troffset - querylength) + tstart;
    trend = (trdiagonal - troffset - querylength) + tend;
    
    debug10(printf("Comparing tr_endpoints %d..%d with exon %d..%d",trstart,trend,exstart,exend));

    if (exend <= trstart) {
      exstart = exonbounds[exoni];
      exoni += 1;
      debug10(printf(" => Advancing exoni to be %d\n",exoni));

    } else {
      /* tstart begins in this exon */
      exonpos = trstart - exstart;
      exonlength = exend - exstart;
      /* Need to subtract 1 from from exonstarts[exoni] only for geneplus */
      univdiagonal = chroffset + (Univcoord_T) (exonstarts[exoni] - 1/*1-based*/ + exonpos - tstart + querylength); /* geneplus */
      debug10(printf(" => Exonpos is %d = trstart %d - exstart %d, out of an exonlength of %d\n",
		     exonpos,trstart,exstart,exonlength));
      debug10(printf("univdiagonal %u = chroffset %u + exonstart %d-1 + exonpos %d - tstart %d + querylength %d\n",
		     univdiagonal,chroffset,exonstarts[exoni],exonpos,tstart,querylength));

      if (between_junction != NULL) {
#ifdef DEBUG10
	printf("Adding junction: "); Junction_print(between_junction); printf("\n");
#endif
	junctions = Listpool_push(junctions,listpool,(void *) between_junction
				  listpool_trace(__FILE__,__LINE__));
      }

      if (trend <= exend) {
	/* tinterval is entirely within this exon */
	if (tr_junctions == NULL) {
	  between_junction = (Junction_T) NULL;
	} else {
	  between_junction = Junction_copy((Junction_T) List_head(tr_junctions),pathpool);
	}

	debug10(printf(">>Output: univdiagonal %u, %d..%d plus transcriptome junction\n",
		       univdiagonal,tstart,tend));
	endpoints = Intlistpool_push(endpoints,intlistpool,tend
				     intlistpool_trace(__FILE__,__LINE__));
	univdiagonals = Univcoordlistpool_push(univdiagonals,univcoordlistpool,univdiagonal
					       univcoordlistpool_trace(__FILE__,__LINE__));
	nmismatches = Intlistpool_push(nmismatches,intlistpool,Intlist_head(tr_nmismatches)
				       intlistpool_trace(__FILE__,__LINE__));
	ref_nmismatches = Intlistpool_push(ref_nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));

	/* Advance transcript information */
	trdiagonals = Uintlist_next(trdiagonals);
	tr_endpoints = Intlist_next(tr_endpoints);
	tr_nmismatches = Intlist_next(tr_nmismatches);
	tr_junctions = List_next(tr_junctions);
	tstart = tend;
	if (Intlist_next(tr_endpoints) != NULL) {
	  tend = Intlist_second_value(tr_endpoints);
	}

      } else {
	/* tinterval past this exon */
	uselength = exend - trstart;
	debug10(printf(">>Output: univdiagonal %u, %d..%d plus splice junction\n",
		       univdiagonal,tstart,tstart + uselength));
	endpoints = Intlistpool_push(endpoints,intlistpool,tstart + uselength
				     intlistpool_trace(__FILE__,__LINE__));
	univdiagonals = Univcoordlistpool_push(univdiagonals,univcoordlistpool,univdiagonal
					       univcoordlistpool_trace(__FILE__,__LINE__));
	if (Intlist_head(tr_nmismatches) == 0) {
	  nmismatches = Intlistpool_push(nmismatches,intlistpool,0
					 intlistpool_trace(__FILE__,__LINE__));
	} else {
	  /* Don't know how to divide a non-zero value of nmismatches between the exons */
	  nmismatches = Intlistpool_push(nmismatches,intlistpool,-1
					 intlistpool_trace(__FILE__,__LINE__));
	  Intlist_head_set(tr_nmismatches,-1);
	}
	ref_nmismatches = Intlistpool_push(ref_nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));

	if (exoni == nexons - 1) {
	  /* tinterval goes to the end of the transcript, so do not add a junction */
	  between_junction = (Junction_T) NULL;
	} else {
	  /* tinterval goes across an intron */
	  splice_distance = exonstarts[exoni + 1] - (exonstarts[exoni] + exonlength); /* gplus */
	  between_junction = Junction_new_splice(splice_distance,sensedir,
						 /*donor_prob*/2.0,/*acceptor_prob*/2.0,pathpool);
	}

	/* Do not advance transcript information */
	Intlist_head_set(tr_endpoints,tstart+uselength);
	exstart = exonbounds[exoni];
	exoni += 1;
      }
    }
  }

  if (univdiagonals == NULL) {
    /* Must be past the end of the transcript */
    Intlistpool_free_list(&endpoints,intlistpool
			  intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
    return (Path_T) NULL;

  } else {
    /* Reverse lists to put them in gplus order */
    endpoints = Intlist_reverse(endpoints);
    univdiagonals = Univcoordlist_reverse(univdiagonals);
    nmismatches = Intlist_reverse(nmismatches);
    ref_nmismatches = Intlist_reverse(ref_nmismatches);
    junctions = List_reverse(junctions);

#ifdef DEBUG10
    printf("endpoints: %s\n",Intlist_to_string(endpoints));
    printf("univdiagonals: %s\n",Univcoordlist_to_string(univdiagonals));
    printf("nmismatches: %s\n",Intlist_to_string(nmismatches));
    printf("junctions: ");
    Junction_print_list(junctions);
    printf("\n");
#endif

    if (Path_endpoints_acceptable_p(endpoints,junctions) == false) {
      debug10(printf("Endpoints not acceptable\n"));
      Intlistpool_free_list(&endpoints,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Univcoordlistpool_free_list(&univdiagonals,univcoordlistpool
				  univcoordlistpool_trace(__FILE__,__LINE__)); /* Allocated by Univcoordlistpool_T */
      Intlistpool_free_list(&nmismatches,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Intlistpool_free_list(&ref_nmismatches,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Junction_list_gc(&junctions,listpool,pathpool);
      return (Path_T) NULL;

    } else {
      /* TODO: Provide initial transcript */
      return Path_create_simple(chrnum,chroffset,chrhigh,
				endpoints,univdiagonals,nmismatches,ref_nmismatches,junctions,
				gplusp,/*genestrand*/0,sensedir,querylength,method,pathpool);
    }
  }
}


Path_T
Trpath_convert_to_path_geneminus (T trpath, int *exonbounds, Chrpos_T *exonstarts, int nexons,
				  Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh,
				  bool tplusp, int sensedir, int querylength,
				  Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
				  Listpool_T listpool, Pathpool_T pathpool, Method_T method) {

  bool gplusp;
  Intlist_T endpoints;
  Univcoordlist_T univdiagonals = NULL;
  Intlist_T nmismatches = NULL, ref_nmismatches = NULL;
  List_T junctions = NULL;

  Univcoord_T univdiagonal;
  Chrpos_T splice_distance;
  Junction_T between_junction = NULL;

  int exoni;
  int exstart, exend;
  int trstart, trend, tstart, tend;
  int exonpos, exonlength, uselength;

  Intlist_T tr_endpoints = trpath->endpoints;
  Uintlist_T trdiagonals = trpath->trdiagonals;
  Intlist_T tr_nmismatches = trpath->nmismatches;
  List_T tr_junctions = trpath->junctions;
  Trcoord_T troffset = trpath->troffset;
  Trcoord_T trdiagonal;

#ifdef DEBUG10
  printf("\nEntered Trpath_convert_to_path_geneminus\n");
  printf("tr_endpoints: %s\n",Intlist_to_string(tr_endpoints));
  printf("trdiagonals: %s\n",Uintlist_to_string(trdiagonals));
  printf("tr_nmismatches: %s\n",Intlist_to_string(tr_nmismatches));
  for (exoni = 0; exoni < nexons; exoni++) {
    printf("exon %d: exonbound %d, exonstart %u\n",exoni,exonbounds[exoni],exonstarts[exoni]);
  }
#endif

  /* tplusp (direction of read along transcript) and gplusp (direction of transcript along genome)
     combine to give plusp (direction of read along genome) */
  gplusp = (tplusp == true) ? false : true;

  /* Initial endpoint */
  tstart = Intlist_head(tr_endpoints);
  tend = Intlist_second_value(tr_endpoints);
  endpoints = Intlistpool_push(NULL,intlistpool,querylength - tstart
			       intlistpool_trace(__FILE__,__LINE__)); /* gminus */

  exoni = 0;
  exstart = 0;
  while (exoni < nexons && trdiagonals != NULL) {
    exend = exonbounds[exoni];
    trdiagonal = Uintlist_head(trdiagonals);
    trstart = (trdiagonal - troffset - querylength) + tstart;
    trend = (trdiagonal - troffset - querylength) + tend;
    
    debug10(printf("Comparing tr_endpoints %d..%d with exon %d..%d",trstart,trend,exstart,exend));

    if (exend <= trstart) {
      exstart = exonbounds[exoni];
      exoni += 1;
      debug10(printf(" => Advancing exoni to be %d\n",exoni));

    } else {
      /* tstart begins in this exon */
      exonpos = trstart - exstart;
      exonlength = exend - exstart;
      /* Need to subtract 1 from from exonstarts[exoni] only for gplus */
      univdiagonal = chroffset + (Univcoord_T) (exonstarts[exoni] - exonpos + tstart); /* gminus */
      debug10(printf(" => Exonpos is %d = trstart %d - exstart %d out of an exonlength of %d\n",
		     exonpos,trstart,exstart,exonlength));
      debug10(printf("univdiagonal %u = chroffset %u + exonstart %d - exonpos %d + tstart %d\n",
		     univdiagonal,chroffset,exonstarts[exoni],exonpos,tstart));

      if (between_junction != NULL) {
#ifdef DEBUG10
	printf("Adding junction: "); Junction_print(between_junction); printf("\n");
#endif
	junctions = Listpool_push(junctions,listpool,(void *) between_junction
				  listpool_trace(__FILE__,__LINE__));
      }

      if (trend <= exend) {
	/* tinterval is entirely within this exon */
	if (tr_junctions == NULL) {
	  between_junction = (Junction_T) NULL;
	} else {
	  between_junction = Junction_copy((Junction_T) List_head(tr_junctions),pathpool);
	  debug10(printf("Inserts advance tend from %d to %d\n",tend,tend+Junction_ninserts(between_junction)));
	  tend += Junction_ninserts(between_junction);
	}

	debug10(printf(">>Output: univdiagonal %u, %d..%d plus transcriptome junction\n",
		       univdiagonal,querylength - tend,querylength - tstart));
	endpoints = Intlistpool_push(endpoints,intlistpool,querylength - tend
				     intlistpool_trace(__FILE__,__LINE__));
	univdiagonals = Univcoordlistpool_push(univdiagonals,univcoordlistpool,univdiagonal
					       univcoordlistpool_trace(__FILE__,__LINE__));
	nmismatches = Intlistpool_push(nmismatches,intlistpool,Intlist_head(tr_nmismatches)
				       intlistpool_trace(__FILE__,__LINE__));
	ref_nmismatches = Intlistpool_push(ref_nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));

	/* Advance transcript information */
	tr_endpoints = Intlist_next(tr_endpoints);
	trdiagonals = Uintlist_next(trdiagonals);
	tr_nmismatches = Intlist_next(tr_nmismatches);
	tr_junctions = List_next(tr_junctions);
	tstart = tend;
	if (Intlist_next(tr_endpoints) != NULL) {
	  tend = Intlist_second_value(tr_endpoints);
	}

      } else {
	/* tinterval past this exon */
	uselength = exend - trstart;
	debug10(printf(">>Output: univdiagonal %u, %d..%d\n",
		       univdiagonal,querylength - (tstart + uselength),querylength - tstart));
	endpoints = Intlistpool_push(endpoints,intlistpool,querylength - (tstart + uselength)
				     intlistpool_trace(__FILE__,__LINE__));
	univdiagonals = Univcoordlistpool_push(univdiagonals,univcoordlistpool,univdiagonal
					       univcoordlistpool_trace(__FILE__,__LINE__));
	if (Intlist_head(tr_nmismatches) == 0) {
	  nmismatches = Intlistpool_push(nmismatches,intlistpool,0
					 intlistpool_trace(__FILE__,__LINE__));
	} else {
	  /* Don't know how to divide a non-zero value of nmismatches between the exons */
	  nmismatches = Intlistpool_push(nmismatches,intlistpool,-1
					 intlistpool_trace(__FILE__,__LINE__));
	  Intlist_head_set(tr_nmismatches,-1);
	}
	ref_nmismatches = Intlistpool_push(ref_nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));

	if (exoni == nexons - 1) {
	  /* tinterval goes to the end of the transcript, so do not add a junction */
	  between_junction = (Junction_T) NULL;
	} else {
	  /* tinterval goes across an intron */
	  splice_distance = (exonstarts[exoni] - exonlength) - exonstarts[exoni + 1]; /* gminus */
	  between_junction = Junction_new_splice(splice_distance,sensedir,
					      /*donor_prob*/2.0,/*acceptor_prob*/2.0,pathpool);
	}

	/* Do not advance transcript information */
	Intlist_head_set(tr_endpoints,tstart+uselength);
	exstart = exonbounds[exoni];
	exoni += 1;
      }
    }
  }

  if (univdiagonals == NULL) {
    /* Must be past the end of the transcript */
    Intlistpool_free_list(&endpoints,intlistpool
			  intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
    return (Path_T) NULL;

  } else {
#ifdef DEBUG10
    printf("endpoints: %s\n",Intlist_to_string(endpoints));
    printf("univdiagonals: %s\n",Univcoordlist_to_string(univdiagonals));
    printf("nmismatches: %s\n",Intlist_to_string(nmismatches));
    printf("junctions: ");
    Junction_print_list(junctions);
    printf("\n");
#endif

    if (Path_endpoints_acceptable_p(endpoints,junctions) == false) {
      debug10(printf("Endpoints not acceptable\n"));
      Intlistpool_free_list(&endpoints,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Univcoordlistpool_free_list(&univdiagonals,univcoordlistpool
				  univcoordlistpool_trace(__FILE__,__LINE__)); /* Allocated by Univcoordlistpool_T */
      Intlistpool_free_list(&nmismatches,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Intlistpool_free_list(&ref_nmismatches,intlistpool
			    intlistpool_trace(__FILE__,__LINE__)); /* Allocated by Intlistpool_T */
      Junction_list_gc(&junctions,listpool,pathpool);
      return (Path_T) NULL;

    } else {
      /* TODO: Provide initial transcript */
      return Path_create_simple(chrnum,chroffset,chrhigh,
				endpoints,univdiagonals,nmismatches,ref_nmismatches,junctions,
				gplusp,/*genestrand*/0,sensedir,querylength,method,pathpool);
    }
  }
}



