/*
 * easy_spice - a spice front end
 * Copyright (C) 2001 Routoure Jean-Marc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTIC2ULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "easyspice.h"


/* convert double value to printable text.
 * Two modes:
 * 0: using suffixes
 *    We always try to print 4 significant figures, with one nonzero digit
 *    to the left of the decimal point.
 *    maximum field width: 7 characters
 *
 * 1: use scientific notation, printf %g format.
 *    maximum field width appears to be 10 characters
 */

gchar*
val2txt(double val, int mode)
{
  static char buf[64];
  double aval = fabs(val);
  double sval, asval;
  char suffix;
  int ddigits;

  switch(mode) {
  case 1:
    sprintf(buf, "% .4g", val);
    break;
  case 0:
  default:
    if (1e12 <= aval) {
      suffix = 'T';
      sval = val / 1e12;
    } else if(1e9 <= aval && aval < 1e12) {
      suffix = 'G';
      sval = val / 1e9;
    } else if(1e6 <= aval && aval < 1e9) {
      suffix = 'M';
      sval = val / 1e6;
    } else if(1e3 <= aval && aval < 1e6) {
      suffix = 'K';
      sval = val / 1000;
    } else if(1e-3 <= aval && aval < 1) {
      suffix = 'm';
      sval = val * 1000;
    } else if(1e-6 <= aval && aval < 1e-3) {
      suffix = 'u';
      sval = val * 1e6;
    } else if(1e-9 <= aval && aval < 1e-6) {
      suffix = 'n';
      sval = val * 1e9;
    } else if(1e-12 <= aval && aval < 1e-9) {
      suffix = 'p';
      sval = val * 1e12;
    } else if(1e-15 <= aval && aval < 1e-12) {
      suffix = 'f';
      sval = val * 1e15;
    } else if(DBL_EPSILON < aval && aval < 1e-15) {
      suffix = 'a';
      sval = val * 1e18;
    } else {
      suffix = ' ';
      sval = val;
    }
    asval = fabs(sval);
    if(1.0 <=  asval && asval < 10.0) 
      ddigits = 3;
    else if(10.0 <=  asval && asval < 100.0) 
      ddigits = 2;
    else 
      ddigits = 1;
    sprintf(buf, "% .*f%c", ddigits, sval, suffix);
    break;
  }	
  return buf;
}

gchar*
update_str(gchar *ptr, const gchar *str)
{
  g_free(ptr);
  return g_strdup(str);
}

void
set_str(gchar **dest, const gchar *str) {
  if (str != NULL) {
    g_free(*dest);
    *dest = (gchar*) str;
  }
}

gchar*
get_gtk_entry(GtkWidget *widget,gchar *entry_name)
{
  return (gchar*) gtk_entry_get_text(GTK_ENTRY(lookup_widget (widget, entry_name)));
}


void 
set_gtk_entry(char *text, GtkWidget *widget, char *entry_name)
{
  GtkWidget *tmp;

  tmp=lookup_widget (GTK_WIDGET (widget), entry_name);
  if (text == NULL)
    gtk_entry_set_text(GTK_ENTRY(tmp),"");
  else
    gtk_entry_set_text(GTK_ENTRY(tmp),text);
}

gboolean 
get_gtk_toggle_button(GtkWidget *widget,char *entry_name)
{
  GtkWidget *tmp;
  tmp=lookup_widget (GTK_WIDGET (widget), entry_name);
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(tmp));
}

void 
set_gtk_toggle_button(GtkWidget *widget,char *entry_name)
{
  GtkWidget *tmp;
  tmp=lookup_widget (GTK_WIDGET (widget), entry_name);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp),TRUE);
}

/* read a text file indicated by text_file and put the result */
/* in the GtkTextBuffer buffer */
gboolean
textfile2buffer(gchar *textfile, GtkTextBuffer *buffer)
{
  FILE *file;
  struct stat buf;
  gchar *tmp,*tmp2;
  gint c;

  if  ((file = fopen(textfile, "r"))==NULL) {
    return FALSE;
  }
  if (stat(textfile, &buf)==0) {
    /* why are we reading this byte by byte? --> fread?? */	
    tmp=(gchar *)g_malloc(buf.st_size*sizeof(gchar)+5);
    if (DEBUG)
      printf("textfile2buffer: file=\"%s\"textsize=%ld\n",
	     textfile, buf.st_size);
    
    for (tmp2=tmp; (c=fgetc(file))!=EOF; tmp2++) {
      *tmp2=c;
    }
    *tmp2='\0';
    
    gtk_text_buffer_set_text(buffer, tmp, -1);
    g_free(tmp);
  }
  fclose(file);
  return TRUE;
}


/* -------------- write a txt message to the log GtkText ------------------*/
void 
write_log(gchar *message)
{
  extern EasySpice *easyspice;
  GtkTextBuffer *buffer;
  GtkTextIter buffer_start, buffer_end;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(easyspice->win_main,"text_log")));
  gtk_text_buffer_get_bounds (buffer,&buffer_start,&buffer_end);
  gtk_text_buffer_insert(GTK_TEXT_BUFFER(buffer),&buffer_end,message,-1);
}

/* -------------------------- Clean the log ------------------------------*/
void 
clean_log(EasySpice *es)
{
  GtkTextBuffer *buffer;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(es->win_main,"text_log")));
  gtk_text_buffer_set_text(buffer,"",-1);
}

/* -------------------------- Clean the std output ------------------------*/

void 
clean_std_output(EasySpice *es)
{
  GtkTextBuffer *buffer;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(es->win_main,"text_spice_output")));
  gtk_text_buffer_set_text(buffer,"",-1);
}

/* -------------------------- Clean the error output ------------------------*/

void 
clean_std_err(EasySpice *es)
{
  GtkTextBuffer *buffer;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(es->win_main,"text_spice_error")));
  gtk_text_buffer_set_text(buffer,"",-1);
}


void 
easyspice_init(EasySpice *es)
{
  es->win_main=NULL;
  es->dialog_file=NULL;
  es->dialog_spicelib=NULL;
  es->dialog_analyse=NULL;
  es->dialog_mainquit=NULL;

  es->project=NULL;
  es->file_schematic=NULL;
  es->file_schematic_tmp=NULL;
  es->file_spice_netlist=NULL;
  es->file_spice_simul=NULL;
  es->file_spice_results=NULL;
  es->file_spice_raw=NULL;
  es->file_spice_plot=NULL;
  es->file_spice_loadplot=NULL;
  es->file_spice_error=NULL;
  es->file_easyspice_project=NULL;

  es->current_n_step=0;
  es->current_simul=0;

  /* FIXME: not finished jet */
  es->simulator_pid=-1;
  es->plotcommand_pid=-1;
}

void 
easyspice_filename_update(EasySpice *es)
{
  g_free(es->file_schematic);
  es->file_schematic=g_strdup_printf("%s.sch",es->project);
  g_free(es->file_schematic_tmp);
  es->file_schematic_tmp=g_strdup_printf("%s_tmp.sch",es->project);
  g_free(es->file_spice_netlist);
  es->file_spice_netlist=g_strdup_printf("%s.net",es->project);
  g_free(es->file_spice_simul);
  es->file_spice_simul=g_strdup_printf("%s.cir",es->project);
  g_free(es->file_spice_results);
  es->file_spice_results=g_strdup_printf("%s.txt.res",es->project);
  g_free(es->file_spice_raw);
  es->file_spice_raw=g_strdup_printf("%s_rawspice.raw",es->project);
  g_free(es->file_spice_plot);
  es->file_spice_plot=g_strdup_printf("%s_plot.cir",es->project);
  g_free(es->file_spice_loadplot);
  es->file_spice_loadplot=g_strdup_printf("%s_loadplot.cir",es->project);
  g_free(es->file_spice_error);
  es->file_spice_error=g_strdup_printf("%s.err",es->project);
  g_free(es->file_easyspice_project);
  es->file_easyspice_project=g_strdup_printf("%s.easyspice",es->project);
}

void
spice_variables_init(SpiceVariables *sv)
{
  sv->dc_source1 = NULL;
  sv->dc_start1 = NULL;
  sv->dc_stop1 = NULL;
  sv->dc_inc1 = NULL;
  sv->dc_source2 = NULL;
  sv->dc_start2 = NULL;
  sv->dc_stop2 = NULL;
  sv->dc_inc2 = NULL;
  sv->ac_points = NULL;
  sv->ac_fmin = NULL;
  sv->ac_fmax = NULL;
  sv->noise_node = NULL;
  sv->noise_source = NULL;
  sv->noise_points = NULL;
  sv->noise_fmin = NULL;
  sv->noise_fmax = NULL;
  sv->tran_tstop = NULL;
  sv->tran_tstep = NULL;
  sv->tran_tmax = NULL;
  sv->tran_tstart = NULL;
  sv->tf_output = NULL;
  sv->tf_source = NULL;
  sv->param_name = NULL;
  sv->param_param = NULL;
  sv->param_start = NULL;
  sv->param_stop = NULL;
  sv->param_step = NULL;
  sv->param_values = NULL;
  sv->plot_pre = NULL;
  sv->plot_xvalue = NULL;
  sv->plot_yvalues = NULL;
}

void 
spice_variables_default(SpiceVariables *sv)
{
  sv->op_show_netlist = FALSE;
  sv->op_show_models = FALSE;
  sv->op_show_results_schematic = FALSE;

  sv->dc_source1 = update_str(sv->dc_source1, "");
  sv->dc_start1 = update_str(sv->dc_start1, "0");
  sv->dc_stop1 = update_str(sv->dc_stop1, "5");
  sv->dc_inc1 = update_str(sv->dc_inc1, "0.1");;
  sv->dc_source2 = update_str(sv->dc_source2, "");
  sv->dc_start2 = update_str(sv->dc_start2, "");
  sv->dc_stop2 = update_str(sv->dc_stop2, "");
  sv->dc_inc2 = update_str(sv->dc_inc2, "");

  sv->ac_type = SWEEP_DEC;
  sv->ac_points = update_str(sv->ac_points, "10");
  sv->ac_fmin = update_str(sv->ac_fmin, "10");
  sv->ac_fmax = update_str(sv->ac_fmax, "100000");

  sv->noise_node = update_str(sv->noise_node, "");
  sv->noise_source = update_str(sv->noise_source, "");
  sv->noise_type = SWEEP_DEC;
  sv->noise_points = update_str(sv->noise_points, "10");
  sv->noise_fmin = update_str(sv->noise_fmin, "1Meg");
  sv->noise_fmax = update_str(sv->noise_fmax, "10G");

  sv->tran_tstop = update_str(sv->tran_tstop, "100us");
  sv->tran_tstep = update_str(sv->tran_tstep, "100ns");
  sv->tran_tmax = update_str(sv->tran_tmax, "");
  sv->tran_tstart = update_str(sv->tran_tstart, "");

  sv->tf_output = update_str(sv->tf_output, "");
  sv->tf_source = update_str(sv->tf_source, "");

  sv->param_activ = FALSE;
  sv->param_name = update_str(sv->param_name, "");
  sv->param_param = update_str(sv->param_param, "");
  sv->param_type = SWEEP_LIN;
  sv->param_start = update_str(sv->param_start, "");
  sv->param_stop = update_str(sv->param_stop, "");
  sv->param_step = update_str(sv->param_step, "");
  sv->param_values = update_str(sv->param_values, "");

  sv->plot_pre = update_str(sv->plot_pre, "");
  sv->plot_xdefault = TRUE;
  sv->plot_xvalue = update_str(sv->plot_xvalue, "");
  sv->plot_xlogscale = FALSE;
  sv->plot_xlinscale = FALSE;
  sv->plot_yvalues = update_str(sv->plot_yvalues, "");
  sv->plot_ylogscale = FALSE;
  sv->plot_autoupdate =FALSE;
}

void 
spice_variables_set_entries(GtkWidget *win, SpiceVariables *sv)
{
  GtkTextBuffer *buffer;
  if (sv->op_show_netlist != FALSE)
    set_gtk_toggle_button(win, "op_listing");
  if (sv->op_show_models != FALSE)
    set_gtk_toggle_button(win, "op_param_view");
  if (sv->op_show_results_schematic != FALSE)
    set_gtk_toggle_button(win, "op_op_view_schem");

  set_gtk_entry(sv->dc_source1, win, "dc_source1");
  set_gtk_entry(sv->dc_start1, win, "dc_start1");
  set_gtk_entry(sv->dc_stop1, win, "dc_stop1");
  set_gtk_entry(sv->dc_inc1, win, "dc_inc1");
  set_gtk_entry(sv->dc_source2, win, "dc_source2");
  set_gtk_entry(sv->dc_start2, win, "dc_start2");
  set_gtk_entry(sv->dc_stop2, win, "dc_stop2");
  set_gtk_entry(sv->dc_inc2, win, "dc_inc2");

  if (sv->ac_type == SWEEP_DEC) 
    set_gtk_toggle_button(win, "ac_dec");
  if (sv->ac_type == SWEEP_LIN)
    set_gtk_toggle_button(win, "ac_lin");
  if (sv->ac_type == SWEEP_OCT)
    set_gtk_toggle_button(win, "ac_oct");

  set_gtk_entry(sv->ac_points, win, "ac_nombre");
  set_gtk_entry(sv->ac_fmin, win, "ac_fmin");
  set_gtk_entry(sv->ac_fmax, win, "ac_fmax");

  set_gtk_entry(sv->noise_node, win, "noise_node");
  set_gtk_entry(sv->noise_source, win, "noise_source");
  if (sv->noise_type == SWEEP_DEC)
    set_gtk_toggle_button(win, "noise_dec"); 
  if (sv->noise_type == SWEEP_LIN)
    set_gtk_toggle_button(win, "noise_lin"); 
  if (sv->noise_type == SWEEP_OCT)
    set_gtk_toggle_button(win, "noise_oct");

  set_gtk_entry(sv->noise_points, win, "noise_nombre");
  set_gtk_entry(sv->noise_fmin, win, "noise_fmin");
  set_gtk_entry(sv->noise_fmax, win, "noise_fmax");

  set_gtk_entry(sv->tran_tstop, win, "tran_tstop");
  set_gtk_entry(sv->tran_tstep, win, "tran_tstep");
  set_gtk_entry(sv->tran_tmax, win, "tran_tmax");
  set_gtk_entry(sv->tran_tstart, win, "tran_tstart");

  set_gtk_entry(sv->tf_output, win, "tf_output");
  set_gtk_entry(sv->tf_source, win, "tf_source");

  if (sv->param_activ != FALSE)
    set_gtk_toggle_button(win, "param_enable");
  set_gtk_entry(sv->param_name, win, "param_name");
  set_gtk_entry(sv->param_param, win, "param_param");
  if (sv->param_type == SWEEP_DEC)
    set_gtk_toggle_button(win, "param_dec"); 
  if (sv->param_type == SWEEP_LIN)
    set_gtk_toggle_button(win, "param_lin");
  if (sv->param_type == SWEEP_LIST)
    set_gtk_toggle_button(win, "param_list"); 
  set_gtk_entry(sv->param_start, win, "param_start");
  set_gtk_entry(sv->param_stop, win, "param_stop");
  set_gtk_entry(sv->param_step, win, "param_step");
  set_gtk_entry(sv->param_values, win, "param_values");

  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(lookup_widget(win,"plot_pre_text")));
  gtk_text_buffer_set_text(buffer, sv->plot_pre, -1);

  if (sv->plot_xdefault != FALSE)
    set_gtk_toggle_button(win, "plot_xdefault");
  set_gtk_entry(sv->plot_xvalue, win, "plot_x");
  if (sv->plot_xlogscale != FALSE)
    set_gtk_toggle_button(win, "plot_xlog_scale");
  if (sv->plot_xlinscale != FALSE)
    set_gtk_toggle_button(win, "plot_xlin_scale");
  set_gtk_entry(sv->plot_yvalues, win, "plot_y");
  if (sv->plot_ylogscale != FALSE)
    set_gtk_toggle_button(win, "plot_ylog_scale");
  if (sv->plot_autoupdate != FALSE)
    set_gtk_toggle_button(win, "plot_auto_update");
}

void 
spice_variables_get_entries(GtkWidget *win, SpiceVariables *sv)
{
  GtkTextBuffer *buffer;
  GtkTextIter buffer_start, buffer_end;

  sv->op_show_netlist = get_gtk_toggle_button(win, "op_listing");
  sv->op_show_models = get_gtk_toggle_button(win, "op_param_view");
  sv->op_show_results_schematic = get_gtk_toggle_button(win, "op_op_view_schem");

  sv->dc_source1 = update_str(sv->dc_source1, get_gtk_entry(win, "dc_source1"));
  sv->dc_start1 = update_str(sv->dc_start1, get_gtk_entry(win, "dc_start1"));
  sv->dc_stop1 = update_str(sv->dc_stop1, get_gtk_entry(win, "dc_stop1"));
  sv->dc_inc1 = update_str(sv->dc_inc1, get_gtk_entry(win, "dc_inc1"));
  sv->dc_source2 = update_str(sv->dc_source2, get_gtk_entry(win, "dc_source2"));
  sv->dc_start2 = update_str(sv->dc_start2, get_gtk_entry(win, "dc_start2"));
  sv->dc_stop2 = update_str(sv->dc_stop2, get_gtk_entry(win, "dc_stop2"));
  sv->dc_inc2 = update_str(sv->dc_inc2, get_gtk_entry(win, "dc_inc2"));

  if (get_gtk_toggle_button(win, "ac_dec")) sv->ac_type= SWEEP_DEC; 
  if (get_gtk_toggle_button(win, "ac_lin")) sv->ac_type= SWEEP_LIN; 
  if (get_gtk_toggle_button(win, "ac_oct")) sv->ac_type= SWEEP_OCT;

  sv->ac_points = update_str(sv->ac_points, get_gtk_entry(win, "ac_nombre"));
  sv->ac_fmin = update_str(sv->ac_fmin, get_gtk_entry(win, "ac_fmin"));
  sv->ac_fmax = update_str(sv->ac_fmax, get_gtk_entry(win, "ac_fmax"));

  sv->noise_node = update_str(sv->noise_node, get_gtk_entry(win, "noise_node"));
  sv->noise_source = update_str(sv->noise_source, get_gtk_entry(win, "noise_source"));
  if (get_gtk_toggle_button(win, "noise_dec")) sv->noise_type=SWEEP_DEC; 
  if (get_gtk_toggle_button(win, "noise_lin")) sv->noise_type=SWEEP_LIN; 
  if (get_gtk_toggle_button(win, "noise_oct")) sv->noise_type=SWEEP_OCT; 

  sv->noise_points = update_str(sv->noise_points, get_gtk_entry(win, "noise_nombre"));
  sv->noise_fmin = update_str(sv->noise_fmin, get_gtk_entry(win, "noise_fmin"));
  sv->noise_fmax = update_str(sv->noise_fmax, get_gtk_entry(win, "noise_fmax"));

  sv->tran_tstop = update_str(sv->tran_tstop, get_gtk_entry(win, "tran_tstop"));
  sv->tran_tstep = update_str(sv->tran_tstep, get_gtk_entry(win, "tran_tstep"));
  sv->tran_tmax = update_str(sv->tran_tmax, get_gtk_entry(win, "tran_tmax"));
  sv->tran_tstart = update_str(sv->tran_tstart, get_gtk_entry(win, "tran_tstart"));

  sv->tf_output = update_str(sv->tf_output, get_gtk_entry(win, "tf_output"));
  sv->tf_source = update_str(sv->tf_source, get_gtk_entry(win, "tf_source"));

  sv->param_activ = get_gtk_toggle_button(win, "param_enable");
  sv->param_name = update_str(sv->param_name, get_gtk_entry(win, "param_name"));
  sv->param_param = update_str(sv->param_param, get_gtk_entry(win, "param_param"));
  if (get_gtk_toggle_button(win, "param_dec")) sv->param_type=SWEEP_DEC; 
  if (get_gtk_toggle_button(win, "param_lin")) sv->param_type=SWEEP_LIN; 
  if (get_gtk_toggle_button(win, "param_list")) sv->param_type=SWEEP_LIST; 
  sv->param_start = update_str(sv->param_start, get_gtk_entry(win, "param_start"));
  sv->param_stop = update_str(sv->param_stop, get_gtk_entry(win, "param_stop"));
  sv->param_step = update_str(sv->param_step, get_gtk_entry(win, "param_step"));
  sv->param_values = update_str(sv->param_values, get_gtk_entry(win, "param_values"));

  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(lookup_widget(win,"plot_pre_text")));
  gtk_text_buffer_get_bounds(buffer, &buffer_start, &buffer_end);
  g_free(sv->plot_pre);
  sv->plot_pre=gtk_text_buffer_get_text(buffer,&buffer_start, &buffer_end, TRUE);

  sv->plot_xdefault = get_gtk_toggle_button(win, "plot_xdefault");
  sv->plot_xvalue = update_str(sv->plot_xvalue,get_gtk_entry(win, "plot_x"));
  sv->plot_xlogscale = get_gtk_toggle_button(win, "plot_xlog_scale");
  sv->plot_xlinscale = get_gtk_toggle_button(win, "plot_xlin_scale");
  sv->plot_yvalues = update_str(sv->plot_yvalues, get_gtk_entry(win, "plot_y"));
  sv->plot_ylogscale = get_gtk_toggle_button(win, "plot_ylog_scale");
  sv->plot_autoupdate = get_gtk_toggle_button(win, "plot_auto_update");
}

void 
spice_variables_delete(SpiceVariables *sv)
{

}

void
readspice_value(gchar **dest, gchar *s1, gchar *s2, gchar *value)
{
  if (strcmp(s1,s2) == 0) {
    g_free(*dest);
    *dest = g_strdup(value);
  }
}

void
textviewer_init(Textviewer *tv)
{
  tv->dialog=NULL;
}

void 
geda_settings_init(GedaSettings *geda)
{
  geda->dialog=NULL;
  geda->include_dir=NULL;
  geda->gnetlist_command_line=NULL;
  geda->attr=NULL;
  geda->gschem_textsize=NULL;
  geda->gschem_gridsize=NULL;
}

void 
geda_settings_default(GedaSettings *geda)
{
  geda->include_cwd=TRUE;
  geda->include_dir=update_str(geda->include_dir, "");
  geda->gnetlist_command_line=update_str(geda->gnetlist_command_line, "gnetlist -g spice -o");
  geda->pinlabel_attr=TRUE;
  geda->attr=update_str(geda->attr, "");
  geda->gschem_textsize=update_str(geda->gschem_textsize, "8");
  geda->gschem_gridsize=update_str(geda->gschem_gridsize, "100");
  geda->gnetlist_priority_node=TRUE;
  geda->gnetlist_hierarchy_allow=TRUE;
}

void 
geda_settings_delete(GedaSettings *geda)
{
  g_free(geda->include_dir);
  g_free(geda->gnetlist_command_line);
  g_free(geda->attr);
  g_free(geda->gschem_textsize);
  g_free(geda->gschem_gridsize);
}

void 
geda_settings_set_entries(GtkWidget *dialog, GedaSettings *geda)
{
  set_gtk_entry(geda->include_dir, dialog, "include_directory_entry");
  set_gtk_entry(geda->gnetlist_command_line, dialog, "gnetlist_command_line_entry");
  set_gtk_entry(geda->attr, dialog, "geda_labels_entry");
  set_gtk_entry(geda->gschem_textsize, dialog, "text_size_entry");
  set_gtk_entry(geda->gschem_gridsize, dialog, "snap_size_entry");
  if (geda->include_cwd == TRUE)
    set_gtk_toggle_button(dialog, "includecwd_yes_radiobutton");
  else
    set_gtk_toggle_button(dialog, "includecwd_no_radiobutton");
  if (geda->pinlabel_attr == TRUE)
    set_gtk_toggle_button(dialog, "pinlabel_yes_radiobutton");
  else
    set_gtk_toggle_button(dialog, "pinlabel_no_radiobutton");
  if (geda->gnetlist_priority_node == TRUE)
    set_gtk_toggle_button(dialog, "netpriority_nodes_radiobutton");
  else
    set_gtk_toggle_button(dialog, "netpriority_labels_radiobutton");
  if (geda->gnetlist_hierarchy_allow == TRUE)
    set_gtk_toggle_button(dialog, "hierarchy_yes_radiobutton");
  else
    set_gtk_toggle_button(dialog, "hierarchy_no_radiobutton");
}

void 
geda_settings_get_entries(GtkWidget *dialog, GedaSettings *geda)
{
  geda->include_dir=update_str(geda->include_dir, 
			       get_gtk_entry(dialog, "include_directory_entry"));
  geda->gnetlist_command_line=update_str(geda->gnetlist_command_line, 
			       get_gtk_entry(dialog, "gnetlist_command_line_entry"));
  geda->attr=update_str(geda->attr, get_gtk_entry(dialog, "geda_labels_entry"));
  geda->gschem_textsize=update_str(geda->gschem_textsize, 
				 get_gtk_entry(dialog, "text_size_entry"));
  geda->gschem_gridsize=update_str(geda->gschem_gridsize, 
				   get_gtk_entry(dialog, "snap_size_entry"));

  printf("%d",get_gtk_toggle_button(dialog, "includecwd_yes_radiobutton") );
  if (get_gtk_toggle_button(dialog, "includecwd_yes_radiobutton") == TRUE) 
    geda->include_cwd=TRUE; 
  else 
    geda->include_cwd=FALSE;
  if (get_gtk_toggle_button(dialog, "pinlabel_yes_radiobutton") == TRUE)
    geda->pinlabel_attr=TRUE;
  else
    geda->pinlabel_attr=FALSE;
  if (get_gtk_toggle_button(dialog, "netpriority_nodes_radiobutton") == TRUE)
    geda->gnetlist_priority_node=TRUE;
  else
    geda->gnetlist_priority_node=FALSE;
  if (get_gtk_toggle_button(dialog, "hierarchy_yes_radiobutton") == TRUE)
    geda->gnetlist_hierarchy_allow=TRUE;
  else
    geda->gnetlist_hierarchy_allow=FALSE;
}

void 
commands_init(ExternalProgramms *prog)
{
  prog->dialog=NULL;
  prog->editor=NULL;
  prog->plot_command=NULL;
  prog->spice_command=NULL;
  prog->browser=NULL;
  prog->html_files=NULL;
  prog->spicehelp_command=NULL;
}

void 
commands_set_default(ExternalProgramms *prog)
{
  prog->editor=update_str(prog->editor, "xemacs");
  prog->plot_command=update_str(prog->plot_command, "ngspice");
  prog->spice_command=update_str(prog->spice_command, "ngspice");
  prog->browser=update_str(prog->browser, "firefox");
  prog->html_files=update_str(prog->html_files,
			      "file:///usr/local/doc/spice/index.html");
  prog->spicehelp_command=update_str(prog->spicehelp_command, "nghelp");
}

void 
commands_delete(ExternalProgramms *prog)
{
  g_free(prog->editor);
  g_free(prog->plot_command);
  g_free(prog->spice_command);
  g_free(prog->browser);
  g_free(prog->html_files);
  g_free(prog->spicehelp_command);
}

void 
commands_set_entries(GtkWidget *dialog, ExternalProgramms *prog)
{
  set_gtk_entry(prog->editor, dialog, "texteditor_entry");
  set_gtk_entry(prog->plot_command, dialog, "plot_command_entry");
  set_gtk_entry(prog->spice_command, dialog, "simulator_command_entry");
  set_gtk_entry(prog->browser, dialog, "browser_entry");
  set_gtk_entry(prog->html_files, dialog, "html_index_entry");
  set_gtk_entry(prog->spicehelp_command, dialog, "help_command_entry");
}

void 
commands_get_entries(GtkWidget *dialog, ExternalProgramms *prog)
{
  prog->editor=update_str(prog->editor,
			  get_gtk_entry(dialog, "texteditor_entry"));
  prog->plot_command=update_str(prog->plot_command,
				get_gtk_entry(dialog, "plot_command_entry"));
  prog->spice_command=update_str(prog->spice_command,
				 get_gtk_entry(dialog, "simulator_command_entry"));
  prog->browser=update_str(prog->browser,
			   get_gtk_entry(dialog, "browser_entry"));
  prog->html_files=update_str(prog->html_files,
			      get_gtk_entry(dialog, "html_index_entry"));
  prog->spicehelp_command=update_str(prog->spicehelp_command,
				     get_gtk_entry(dialog, "help_command_entry"));
}

void
postprocessing_init(PostProcessing *postproc)
{
  postproc->dialog=NULL;
  postproc->commands=g_strdup("");
}

void
postprocessing_set_entries(PostProcessing *postproc)
{
  GtkTextBuffer *buffer;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(postproc->dialog
						 ,"textview")));
  gtk_text_buffer_set_text(buffer,postproc->commands,-1);
}

void
postprocessing_get_entries(PostProcessing *postproc)
{
  GtkTextBuffer *buffer;
  GtkTextIter start, end;
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW
				  (lookup_widget(postproc->dialog
						 ,"textview")));
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  g_free(postproc->commands);
  postproc->commands=gtk_text_buffer_get_text(buffer, &start, &end, TRUE);
}

void
postprocessing_delete(PostProcessing *postproc) 
{
  g_free(postproc->commands); 
  postproc->commands=g_strdup("");
}

void
spicelibrary_init(SpiceLibrary *spicelib) 
{
  spicelib->dialog=NULL;
  spicelib->libs=NULL;
  spicelib->store=NULL;
}

void
spicelibrary_delete(SpiceLibrary *spicelib)
{
  g_list_foreach(spicelib->libs, (GFunc) free_element, NULL);
  g_list_free(spicelib->libs);
  spicelib->libs=NULL;
}


/* -------------------------------------------------------------------*/
/* convert a sci notation k to e3 and so on                           */
/* ------------------------------------------------------------------ */
double
spice2double(gchar *sp_value)
{
  gchar *unit;
  double value;
  gint token;
  
  /* get enough memory for sp_value */
  unit = g_strdup(sp_value);

  token = sscanf(sp_value," %lf%s",&value, unit);
  
  if (token == 0) { /* no value given */
    printf(_("spice2double: no value given [%s]\n"),sp_value);
  }
  else if (token == 1) { /* value is allready sciformat */
    /* void */
  }
  else if (token == 2) { /* unit needs to be converted */
    switch (unit[0]) {
    case 'f':
    case 'F':
      value *= 1e-15;
      break;
    case 'p':
    case 'P':
      value *= 1e-12;
      break;
    case 'n':
    case 'N':
      value *= 1e-9;
      break;
    case 'u':
    case 'U':
      value *= 1e-6;
      break;
    case 'm':
      value *=1e-3;
      break;
    case 'M':
      if ((unit[1] =='e' && unit[2] == 'g')
	  || (unit[1] == 'E' && unit[2] == 'G'))
	value *= 1e+6;
      else
	value *= 1e-3;
      break;
    case 'k':
    case 'K':
      value *= 1e+3;
      break;
    case 'g':
    case 'G':
      value *= 1e9;
      break;
    default:
      printf(_("spice2double: strange unit [%s] in value [%s]\n"),unit,sp_value);
    }
  }
  g_free(unit);

  return value;
}

void
project_set_title(EasySpice *es) 
{
  gchar *window_title;

  window_title = g_strdup_printf("%s - easy_spice", es->project);
  gtk_window_set_title (GTK_WINDOW (es->win_main), window_title);
  g_free(window_title);
}

void 
textfile_view(Textviewer *tv, gchar *filename)
{
  GtkTextBuffer *buffer;
  gchar *buf;
  
  if (tv->dialog == NULL) {
    tv->dialog = create_txt_viewer(tv);
  }
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(lookup_widget(tv->dialog,"gtk_txt_view")));
  if (textfile2buffer(filename, buffer) == TRUE) {
    gtk_label_set_text (GTK_LABEL(lookup_widget(tv->dialog,"label_filename")),
			filename);
    gtk_widget_show(tv->dialog);
    gtk_window_present(GTK_WINDOW(tv->dialog));
  }
  else {
    buf=g_strdup_printf(_("Error reading text file %s\n"), filename);
    write_log(buf);
    g_free(buf);
    gtk_widget_destroy(tv->dialog);
    tv->dialog = NULL;
  }
}

/* switch on/off the "LED" in the main windows */
void 
led_ok(GtkWidget *win)
{
  gtk_widget_show(lookup_widget (win, "pixmap_log_ok"));
  gtk_widget_hide(lookup_widget (win, "pixmap_log_false"));
}

void 
led_false(GtkWidget *win)
{
  gtk_widget_hide(lookup_widget (win, "pixmap_log_ok"));
  gtk_widget_show(lookup_widget (win, "pixmap_log_false"));
}

void 
led_off(GtkWidget *win)
{
  gtk_widget_hide(lookup_widget (win, "pixmap_log_ok"));
  gtk_widget_hide(lookup_widget (win, "pixmap_log_false"));
}

void
keyvalue_print_list(GList *item)
{
  for (item=g_list_first(item);item != NULL; item=g_list_next(item)) {
    printf("  key=\"%s\", value=\"%s\"\n",
	   ((KeyValue*)(item->data))->key, ((KeyValue*)(item->data))->value);
  }
}

gint
keyvalue_compare_key(KeyValue *kv1, gchar *key)
{
  gint len;
  len=strlen(kv1->key);
  if (strlen(key) > len)
    len=strlen(key);

  return g_ascii_strncasecmp(kv1->key, key, len);
}

gint
spicedevice_compare_name(SpiceDevice *sd, gchar *name)
{
  gint len;
  len=strlen(sd->name);
  if (strlen(name) > len)
    len=strlen(name);

  return g_ascii_strncasecmp(sd->name, name, len);
}
