/**
 * @brief Low level backlight driver for PMU controlled backlights.
 *
 * This module contains the low level driver to control a LCD backlight
 * via the PMU. The low level driver overloads methods of class backlight.
 * 
 * To make this driver work under kernel versions > 2.6.18 the kernel
 * option CONFIG_PMAC_BACKLIGHT_LEGACY must be set. This option includes
 * some old ioctls for /dev/pmu that are needed here. For older kernels
 * it will work out of the box.
 *
 * Every frame buffer driver could register a backlight controller,
 * but only the right one is accepted. Verification is done by checking
 * OpenFirmware's property "backlight-control" in register_backlight_controller().
 * ioctl(base->fd_pmu, PMU_IOC_GET_BACKLIGHT, &val);  returns -ENODEV if no
 * backlight device is registered. This is used to check if a backlight
 * controller is installed. If no backlight controller is available the
 * initialization of this driver will fail.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/driver_backlight_pmu.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 *
 * @todo  This module accesses the PMU functions of module pmac. The
 *        better way would be to put all PMU and ADB functions in a
 *        seperate library file and include that.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/pmu.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "module_pmac.h"
#include "class_backlight.h"
#include "class_config.h"
#include "driver_backlight_pmu.h"
#include "support.h"

/**
 * @brief filehandle to access the PMU
 *
 * This filehandle is needed to communicate with the PMU. The PMU
 * controlls the LCD backlight in various Apple machines.
 */
static int pmu_filehandle;

/**
 * @brief Exit function of the driver - cleans up all ressources
 *
 * This function will be called when the work is done and the driver
 * should be unloaded. It frees all allocated ressources.
 */
void
driver_backlight_pmu_exit ()
{
	if (pmu_filehandle != -1)
		close (pmu_filehandle);
}

/**
 * @brief Get the current brightness level from the device
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the current brightness level of the LCD backlight.
 *
 * @return  current brightness level or -1 on error
 */
int
pmubl_get_brightness ()
{
	long val = 0;
		
	if ((ioctl(pmu_filehandle, PMU_IOC_GET_BACKLIGHT, &val)) == -1)
		return -1;

	return (int) val;
}

/**
 * @brief Get the maximum brightness level the device supports
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the maximum brightness level the LCD backlight
 * device supports. In this case it is a fixed value of 15 because
 * the amount of brightness steps for PMU controlled backlight
 * devices is har coded in the kernel. The graphics hardware would
 * be able to support more than 15 steps but the kernel programmers
 * didn't reflect in that.
 *
 * @return  maximum supported brightness level of the device.
 */
int
pmubl_get_brightness_max ()
{
	return PMUBRIGHTNESSMAX;
}

/**
 * @brief Change the current brightness level.
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * This function sets a new brightness level. The given level must be
 * valid for this device, that means it must be lower than the level
 * pmubl_get_brightness_max() returns. No further check is done on that.
 *
 * @param  val   new brightness level
 */
void
pmubl_set_brightness (int val)
{
	if (val != -1)
		ioctl(pmu_filehandle, PMU_IOC_SET_BACKLIGHT, &val);
}

static struct driver_backlight driver_backlight_pmu = {
	.name               = N_("PMU Backlight Driver"),
	.get_brightness     = pmubl_get_brightness,
	.get_brightness_max = pmubl_get_brightness_max,
	.set_brightness     = pmubl_set_brightness,
	.driver_exit        = driver_backlight_pmu_exit,
};

/**
 * @brief Constructor of a PMU backlight driver object
 *
 * This function probes for a PMU and if the LCD backlight could
 * be controlled with it. If so, an initialized driver object is
 * returned, otherwise NULL.
 *
 * @return  Initializes backlight driver object or NULL
 */
struct driver_backlight *
driver_backlight_pmu_init ()
{
	int pmu_version = 0, fd;
	char *device;
	long DUMMY = 0;

	device = config_get_string ("MODULE PMAC", "Device_PMU", DEFAULT_PMU);
	if ((check_path (device, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((pmu_filehandle = open(device, O_RDWR)) < 0) {
			print_msg (PBB_ERR, _("Can't open PMU device %s: %s\n"), device, strerror(errno));
			goto out;
		}
	} else goto out;

	device = config_get_string ("MODULE PMAC", "Device_ADB", DEFAULT_ADB);
	if ((check_path (device, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((fd = open(device, O_RDWR)) >= 0) {
			pmu_version = get_pmu_version(fd);
			close (fd);
		} else
			print_msg (PBB_WARN, _("Can't open ADB device %s: %s\n"), device, strerror(errno));
	};
	g_free (device);

	if ((ioctl(pmu_filehandle, PMU_IOC_GET_BACKLIGHT, &DUMMY)) == -1)
		return NULL;

	if (pmu_version > OHARE_PMU)
		ioctl(pmu_filehandle, PMU_IOC_GRAB_BACKLIGHT, &DUMMY);

	return &driver_backlight_pmu;
out:
	g_free (device);
	return NULL;
}

