// Copyright 2010-2016, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package org.mozc.android.inputmethod.japanese.view;

import com.google.common.base.Preconditions;

import android.graphics.Canvas;
import android.graphics.Picture;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.PictureDrawable;

/**
 * This class is very similar to {@link PictureDrawable}, except this class scale to fit
 * to the given bounding box as similar to other drawables.
 *
 */
public class MozcPictureDrawable extends PictureDrawable {
  public MozcPictureDrawable(Picture picture) {
    super(Preconditions.checkNotNull(picture));
  }

  @Override
  public void draw(Canvas canvas) {
    Preconditions.checkArgument(!canvas.isHardwareAccelerated(),
        "MozcPictureDrawable doesn't accept h/w accelerated canvas, "
        + "which doesn't support drawPicture().");
    Picture picture = getPicture();
    if (picture == null) {
      return;
    }
    Rect bounds = getBounds();
    int saveCount = canvas.save();
    try {
      canvas.clipRect(bounds);
      canvas.translate(bounds.left, bounds.top);
      float scaleX = bounds.width() / (float) getIntrinsicWidth();
      float scaleY = bounds.height() / (float) getIntrinsicHeight();
      canvas.scale(scaleX, scaleY);
      canvas.drawPicture(picture);
    } finally {
      canvas.restoreToCount(saveCount);
    }
  }

  @Override
  public ConstantState getConstantState() {
    return new ConstantState() {
      @Override
      public int getChangingConfigurations() {
        return 0;
      }

      @Override
      public Drawable newDrawable() {
        return new MozcPictureDrawable(getPicture());
      }
    };
  }
}
