#include "apt-cache-generator-entry.h"

struct _AptCacheGeneratorEntry {
  gchar *name;
  gchar *version;
  
  gchar *webapp_includes;
  gchar *webapp_name;
  gchar *webapp_domain;
};


AptCacheGeneratorEntry *
apt_cache_generator_entry_new (const gchar *name, const gchar *version)
{
  AptCacheGeneratorEntry *entry;
  
  entry = g_new0 (AptCacheGeneratorEntry, 1);

  entry->name = g_strdup (name);
  entry->version = g_strdup (version);

  entry->webapp_includes = NULL;
  entry->webapp_name = NULL;
  entry->webapp_domain = NULL;
  
  return entry;
}

void
apt_cache_generator_entry_free (AptCacheGeneratorEntry *entry)
{
  g_free (entry->name);
  g_free (entry->version);
  
  if (entry->webapp_includes != NULL)
    {
      g_free (entry->webapp_includes);
    }
  if (entry->webapp_name != NULL)
    {
      g_free (entry->webapp_name);
    }
  if (entry->webapp_domain != NULL)
    {
      g_free (entry->webapp_domain);
    }
  
  g_free (entry);
}

static void
apt_cache_generator_entry_append_webapps_metadata (AptCacheGeneratorEntry *entry,
						   GString *s)
{
  if (entry->webapp_includes)
    {
      g_string_append_printf (s, "Ubuntu-Webapps-Includes: %s\n", entry->webapp_includes);
    }
  
  if (entry->webapp_name)
    {
      g_string_append_printf (s, "Ubuntu-Webapps-Name: %s\n", entry->webapp_name);
    }
  if (entry->webapp_domain)
    {
      g_string_append_printf (s, "Ubuntu-Webapps-Domain: %s\n", entry->webapp_domain);
    }
}

gchar *
apt_cache_generator_entry_serialize (AptCacheGeneratorEntry *entry)
{
  GString *s;
  gchar *serialized;
  
  s = g_string_new ("");
  g_string_append_printf (s, "Package: %s\n", entry->name);
  g_string_append_printf (s, "Version: %s\n", entry->version);
  s = g_string_append (s, "Description: Lorum ipsum dolor sit amet\n");
  
  apt_cache_generator_entry_append_webapps_metadata (entry, s);
  
  serialized = s->str;
  g_string_free (s, FALSE);
  
  return serialized;
}

void
apt_cache_generator_entry_add_webapp_include (AptCacheGeneratorEntry *entry, const gchar *include)
{
  if (entry->webapp_includes == NULL)
    {
      entry->webapp_includes = g_strdup (include);
    }
  else
    {
      gchar *t;
      
      t = entry->webapp_includes;
      entry->webapp_includes = g_strconcat (entry->webapp_includes, ";", include, NULL);
      g_free (t);
    }
}

void
apt_cache_generator_entry_set_webapp_name (AptCacheGeneratorEntry *entry, const gchar *webapp_name)
{
  g_assert (entry->webapp_name == NULL);
  
  entry->webapp_name = g_strdup (webapp_name);
}


void
apt_cache_generator_entry_set_webapp_domain (AptCacheGeneratorEntry *entry, const gchar *webapp_domain)
{
  g_assert (entry->webapp_domain == NULL);
  
  entry->webapp_domain = g_strdup (webapp_domain);
}
